<?php
/**
 * Cart Storage trait for OpenApp Gateway.
 *
 * @package OpenApp_Payment_Gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Trait OPENAPPGW_Trait_Cart_Storage
 *
 * Handles cart persistence in custom database table.
 */
trait OPENAPPGW_Trait_Cart_Storage {

    private function generate_cart_hash(){
        $session_id = $this->get_session_id();
        $cart_contents = null;

        if ( function_exists('WC') && isset(WC()->cart) ) {
            $cart_contents = WC()->cart->get_cart_contents();
        }

        // Include the session_id when generating the hash
        $hash = md5(wp_json_encode($cart_contents) . $session_id);
        return $hash;
    }

    public function store_previous_cart_hash_in_session($cart){
        if ($this->is_request_empty()) {
            return;
        }

        if($this->is_heartbeat()){
            return;
        }

        $new_hash = $this->generate_cart_hash();
        $old_hash = WC()->session->get('previous_cart_hash');

        if ($old_hash !== $new_hash) {
            // Update session with new hash
            WC()->session->set('previous_cart_hash', $new_hash);
        }
    }


    private function is_request_empty() {
        return empty($_REQUEST);
    }

    private function generate_oa_basket_id($length = 10) {
        $bytes = random_bytes(ceil($length / 2));
        return substr(bin2hex($bytes), 0, $length);
    }

    private function get_unique_oa_basket_id(){
        global $wpdb;
        $max_attempts = 3;
        $attempts = 0;

        do {
            $cart_id = $this->generate_oa_basket_id();
            $existing_row = $wpdb->get_row($wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $cart_id));
            $attempts++;

            if ($attempts >= $max_attempts && $existing_row) {
                // Log an error if unable to generate a unique cart_id
                error_log('OA: Unable to generate unique cart_id after ' . $max_attempts . ' attempts.');
                return false;
            }
        } while ($existing_row);

        return $cart_id;
    }

    // asynchronous function (in theory: page can be refreshed faster then function completion)
    public function store_cart_in_db($force_rebuild = false)
    {
        if ($this->is_request_empty() && !$force_rebuild) {
            return;
        }

        $is_internal_api = isset($_SERVER['HTTP_X_WP_INTERNAL']) && sanitize_text_field($_SERVER['HTTP_X_WP_INTERNAL']) == 'true';

        if($is_internal_api){
            return;
        }

        $start_time = microtime(true);

        $new_hash = $this->generate_cart_hash();
        $old_hash = WC()->session->get('previous_cart_hash');

        if(!$force_rebuild && ($old_hash === $new_hash)){
           return;
        }



        global $wpdb;

        $session_id = $this->get_session_id();
        $hashed_session_id = hash('md5', $session_id);
        $applied_coupons = WC()->cart->get_applied_coupons();
        $coupon_data = array();

        foreach ($applied_coupons as $coupon_code) {
            $discount_amount = WC()->cart->get_coupon_discount_amount($coupon_code, WC()->cart->display_cart_ex_tax);
            $coupon_data[] = array(
                'code' => $coupon_code,
                'discount_amount' => $discount_amount,
            );
        }

        if(is_user_logged_in()) {
            $userId = strval(get_current_user_id());
        } else {
            $userId = md5(uniqid(microtime() . wp_rand(), true));
        }

        $my_cart = WC()->cart->get_cart();
        $cart_content_data = $this->get_cart_contents_data($my_cart);
        $cart_content = $this->get_simplified_cart_contents($my_cart);

        $cart_data = array(
            'cart_contents' => $cart_content,
            'coupon_data' => $coupon_data,
            'user_id' => $userId,
            'cart_contents_data' => $cart_content_data
        );

        $cart_data_serialized = wp_json_encode($cart_data);

        $existing_row = $wpdb->get_row($wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_session_id = %s", $hashed_session_id));

        if ($existing_row) {
            $wpdb->update(
                $wpdb->prefix . "oa_woocommerce_persistent_cart",
                array('cart_contents' => $cart_data_serialized, 'cart_expiry' => time() + 60 * 60 * 24 * 30),  // expires in 30 days
                array('cart_session_id' => $hashed_session_id),
                array('%s', '%s', '%d'),
                array('%s')
            );

            if (isset($existing_row->cart_id)) {
                $this->cart_id_to_process = $existing_row->cart_id;
                $this->basket_change_triggered = true;
            }
        } else {
            try {
                $cart_id = $this->get_unique_oa_basket_id();
                // set cart_id on new record creation
                $wpdb->insert(
                    $wpdb->prefix . "oa_woocommerce_persistent_cart",
                    array('cart_id' => $cart_id, 'cart_contents' => $cart_data_serialized, 'cart_session_id' => $hashed_session_id, 'cart_expiry' => time() + 60 * 60 * 24 * 30),  // expires in 30 days
                    array('%s', '%s', '%s', '%d')
                );
            } catch (Exception $e) {
                return;
            }
        }

        // End timing
        $end_time = microtime(true);
        $execution_time = round($end_time - $start_time, 4);
        $this->log_debug("store_cart_in_db: ".$execution_time);

    }

    public function on_shutdown() {

        if ('no' === $this->get_option('basket_sync', 'no')) {
            return;
        }

        if ($this->basket_change_triggered && $this->cart_id_to_process) {
            $uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : 'CLI or unknown';
            $method = isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : 'Unknown method';

            // Check for a POST request applying a coupon
            if ($method === 'POST' && strpos($uri, 'wc-ajax=apply_coupon') !== false) {
                return; // Exit early if applying a coupon
            }

            // Continue with POST requests

            // Check for specific GET requests
            if ($method === 'GET') {
                if (!(strpos($uri, 'removed_item') !== false || strpos($uri, 'undo_item') !== false)) {
                    return; // Exit early for GET requests that do NOT involve removing or undoing items
                }
            }

            remove_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));
            $this->oa_basket_changed($this->cart_id_to_process);
            add_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

            $this->basket_change_triggered = false;
            $this->cart_id_to_process = null;
        }
    }

    private function oa_basket_changed($basketId){
        if(!$basketId){
            return;
        }

        $endpoint = '/v1/basket/change';
        $method = 'POST';
        $url = $this->open_app_url . $endpoint;
        $context = 'openapp_basket_changed';

        $start_time = microtime(true);

        // Retrieve the basket data from your database
        $basket_data = $this->get_basket_data($basketId);

        if (empty($basket_data)) {
            return new WP_Error('basket_not_found', 'Basket not found', array('status' => 404));
        }

        $body = wp_json_encode($basket_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        $setupData = $this->setupRequest($endpoint, $method, $body, $context);
        $response = $this->executeRequest($url, $setupData, $context);

        $end_time = microtime(true);
        $execution_time = round($end_time - $start_time, 4);
        $this->log_debug("oa_basket_changed: ".$execution_time);

    }

}
