<?php
/**
 * Helpers trait for OpenApp Gateway.
 *
 * @package OpenApp_Payment_Gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Trait OPENAPPGW_Trait_Helpers
 *
 * Handles cart data functions, utility helpers, and testing/debugging.
 */
trait OPENAPPGW_Trait_Helpers {

    /**
     * ===========================================================================
     * Cart Data Functions
     * ===========================================================================
     */

    private function get_product_object($item){
       if(is_null($item)){
           return null;
       }

        if ($item['variation_id'] > 0) {
            $_product = wc_get_product($item['variation_id']);
        } else {
            $_product = wc_get_product($item['product_id']);
        }

        return $_product;
    }

    private function get_simplified_cart_contents($my_cart) {
        $simplified_cart_contents = [];

        foreach ($my_cart as $cart_item_key => $cart_item) {
            // Copy the cart item, but exclude the 'data' field
            $simplified_cart_item = $cart_item;
            unset($simplified_cart_item['data']);
            $simplified_cart_contents[$cart_item_key] = $simplified_cart_item;
        }

        return $simplified_cart_contents;
    }

    function filterProductDataProperty($product) {
        $productName = $product->get_name();
        $shippingClass = $product->get_shipping_class();
        $weight = $product->get_weight();
        $needsShipping = $product->needs_shipping();
        $customProduct = array(
            'name' => $productName,
            'shippingClass' => $shippingClass,
            'weight' => $weight,
            'needsShipping' => $needsShipping
        );
        return $customProduct;
    }

    private function get_cart_contents_data($cart_contents) {
        // error_log(wp_json_encode($cart_contents, JSON_PRETTY_PRINT));

        $cart_contents_data = [];

        foreach ($cart_contents as $cart_item_key => $cart_item) {
            if (isset($cart_item['data'])) {
                $cart_contents_data[$cart_item_key] = $this->filterProductDataProperty($cart_item['data']);
            }
        }

        return $cart_contents_data;
    }

    public function get_cart_id_by_session($session_id) {
        global $wpdb;

        if(is_null($session_id)){
            return null;
        }

        // $this->log_debug("SESSION_ID:".$session_id);

        // Hashing session ID
        $hashed_session_id = hash('md5', $session_id);

        // Preparing SQL query
        $cart_id = $wpdb->get_var($wpdb->prepare("SELECT cart_id FROM {$wpdb->prefix}oa_woocommerce_persistent_cart WHERE cart_session_id = %s", $hashed_session_id));


        if(is_null($cart_id)){
            // cart is empty.. @TODO ?
            // return $hashed_session_id;
        }

        return $cart_id;
    }


    private function get_basket_data($cart_id) {
        $cart_contents_record = $this->get_cart_contents_record($cart_id);
        if (empty($cart_contents_record)) {
            return array();
        }

        $products = $this->get_products_from_cart_contents($cart_contents_record['cart_contents']);
        $total = $this->calculate_total_price($products);
        $discounts = $this->get_discounts($cart_contents_record['coupon_data']);
        $deliveryOptions = $this->get_available_shipping_methods($this->supported_country, $total, $cart_contents_record);

        $currency = get_woocommerce_currency();

        $basket_data = array(
            'expiresAt' => gmdate('Y-m-d\TH:i:s\Z', strtotime('+1 day')),
            'price' => array(
                'currency' => $currency,
                'basketValue' => round($total, 2),
                'discounts' => $discounts
            ),
            'deliveryOptions' => $deliveryOptions,
            'products' => $products,
            'loggedUser' => $cart_contents_record['user_id']
        );

        $response = array(
            'id' => $cart_id,
        );
        return array_merge($response, $basket_data);
    }

    private function get_cart_contents_record($cart_id) {
        global $wpdb;

        $cart_data = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $cart_id)
        );

        return is_null($cart_data) ? array() : json_decode($cart_data->cart_contents, true);
    }

    private function calculate_total_price($products) {
        $total_in_cents = 0;

        foreach ($products as $item) {
            // Check if this is WooCommerce cart contents (has line_total and line_tax)
            if (isset($item['line_total']) && isset($item['line_tax'])) {
                // Convert each component to cents separately, then add
                // This maintains precision for each tax/total calculation
                $line_total_cents = $this->convertToCents($item['line_total']);
                $line_tax_cents = $this->convertToCents($item['line_tax']);
                $total_in_cents += ($line_total_cents + $line_tax_cents);
            }
            // Check for your custom linePrice field (already in cents)
            elseif (isset($item['linePrice'])) {
                // Ensure it's an integer to avoid float precision issues
                $total_in_cents += (int) $item['linePrice'];
            }
            // Fallback to line_total only
            elseif (isset($item['line_total'])) {
                $total_in_cents += $this->convertToCents($item['line_total']);
            }
        }

        // Return total in cents (ensure integer)
        return (int) $total_in_cents;
    }


    private function get_products_from_cart_contents($cart_contents) {
        $products = array();
        foreach ($cart_contents as $item) {
            $product_data = $this->create_product_output_array_from_cart($item);
            if (!empty($product_data)) {
                $products[] = $product_data;
            }
        }
        return $products;
    }


    /**
     * @param WP_REST_Request $request
     * @return WP_REST_Response|null
     */
    public function get_qr_code_data(WP_REST_Request $request){
        $cart_id = $request->get_param('cart_id');

        if(empty($cart_id)){
            $cart_data = array(
                'merchant_id' => $this->merchant_id,
                'profile_id' => $this->profile_id,
                'cart_id' => $cart_id,
                'total_value' => 0,
                'currency' => '',
                'unique_products_count' => '',
            );
            return $this->create_secure_response($cart_data);
        }

        // Verify cart ownership (prevents enumeration attacks)
        $cart_ownership = $this->verify_cart_ownership($cart_id);
        if (is_wp_error($cart_ownership)) {
            // For unknown/unauthorized carts, return empty structure gracefully
            $cart_data = array(
                'merchant_id' => $this->merchant_id,
                'profile_id' => $this->profile_id,
                'cart_id' => $cart_id,
                'total_value' => 0,
                'currency' => '',
                'unique_products_count' => '',
            );
            return $this->create_secure_response($cart_data);
        }

        // Decode the cart contents from verified cart data
        $cart_data = json_decode($cart_ownership->cart_contents, true);

        if(empty($cart_data)){
            return null;
        }

        $cart_contents = $cart_data['cart_contents'];

        if(empty($cart_contents)){
            // $this->store_cart_in_db(true);
            // for login - we allow empty cart
            // return null;
        }

        // Initialize values
        $unique_products_count = 0;

        $products = array();
        foreach($cart_contents as $item_key => $item) {
            $product_data = $this->create_product_output_array_from_cart($item);
            if(!empty($product_data)){
                array_push($products, $product_data);
                $unique_products_count++;
            }
        }

        $total_value = $this->calculate_total_price($products);


        // Get currency - assuming all products in the cart have the same currency
        $currency = get_woocommerce_currency();

        $response_data = array(
            'merchant_id' => $this->merchant_id,
            'profile_id' => $this->profile_id,
            'cart_id' => $cart_id,
            'total_value' => $total_value,
            'currency' => $currency,
            'unique_products_count' => $unique_products_count,
        );

        return $this->create_secure_response($response_data);

    }


    /**
     * ===========================================================================
     * Utility Helpers
     * ===========================================================================
     */

    private function get_base_url() {
        return rtrim(site_url(), '/');
    }

    private function openappgw_custom_log($message, $context = ''){
        if ( function_exists( 'openappgw_custom_log' ) ) {
            openappgw_custom_log($message, $context);
        }
    }

    private function get_values_and_types($data) {
        if(is_array($data) || is_object($data)) {
            foreach($data as $key => $value) {
                if(is_array($value) || is_object($value)) {
                    $data[$key] = $this->get_values_and_types($value); // Recursive call for nested array or objects
                } else {
                    $data[$key] = array('value' => $value, 'type' => gettype($value));
                }
            }
        } else {
            $data = array('value' => $data, 'type' => gettype($data));
        }
        return $data;
    }

    private function destroy_user_sessions( $user_id ) {
        $sessions = WP_Session_Tokens::get_instance( $user_id );
        $sessions->destroy_all();
    }

    private function get_session_id(){
        $sessionId = null;
        if (isset(WC()->session)) {
            $sessionId = WC()->session->get_customer_id();
        }

        return $sessionId;
    }

    private function createResponse($data, $status = 200)
    {
        return new WP_REST_Response($data, $status);
    }

    /**
     * Create a WP_REST_Response with security headers to prevent caching.
     * Use this for endpoints that return user-specific/sensitive data.
     *
     * @param mixed $data Response data
     * @param int $status HTTP status code (default 200)
     * @return WP_REST_Response
     */
    private function create_secure_response($data, $status = 200) {
        $response = new WP_REST_Response($data, $status);
        $response->header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
        $response->header('Pragma', 'no-cache');
        $response->header('Expires', '0');
        return $response;
    }

    /**
     * Ensure WooCommerce session is initialized for REST API requests.
     * WooCommerce may not auto-initialize sessions in REST context in some environments (e.g., LocalWP).
     */
    private function ensure_wc_session_loaded() {
        if (!function_exists('WC')) {
            return;
        }

        // If session handler doesn't exist, try to initialize it
        if (!isset(WC()->session) || null === WC()->session) {
            // WooCommerce 3.6.4+ has initialize_session()
            if (method_exists(WC(), 'initialize_session')) {
                WC()->initialize_session();
            }
        }
    }

    /**
     * Authenticate REST request from WordPress cookies.
     * WordPress REST API ignores cookies without nonce for CSRF protection.
     * This manually validates cookies for same-origin requests.
     *
     * CSRF Protection: Requires X-WP-Internal header to prevent cross-origin attacks.
     * Browsers won't send custom headers cross-origin without CORS preflight,
     * which our server won't approve for foreign origins.
     */
    private function authenticate_rest_from_cookie() {
        // CASE 1: Already authenticated (e.g., via nonce or another method)
        // No action needed - wp_get_current_user() will already return the user
        if (is_user_logged_in()) {
            return;
        }

        // CASE 2: No WordPress login cookie present
        // User is either:
        // - A guest (never logged in)
        // - Logged out (cookie was cleared)
        // - On a different domain/port (cookies not sent)
        // Fall through to WC session-based authentication in verify_cart_ownership()
        if (!isset($_COOKIE[LOGGED_IN_COOKIE])) {
            return;
        }

        // CASE 3: Cookie exists - attempt to authenticate
        // WordPress REST API intentionally ignores cookies without nonce (CSRF protection)
        // We manually validate for same-origin requests identified by X-WP-Internal header

        // SSE (Server-Sent Events) exception:
        // EventSource API cannot send custom headers, so we skip header check for SSE
        // Security is maintained via cart ownership verification in verify_cart_ownership()
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- SSE detection only
        $is_sse_request = isset($_GET['use_sse']) && $_GET['use_sse'] === '1';

        if (!$is_sse_request) {
            // CSRF Protection for non-SSE requests:
            // X-WP-Internal header proves request is same-origin (browsers block
            // cross-origin custom headers without CORS preflight approval)
            $internal_header = isset($_SERVER['HTTP_X_WP_INTERNAL']) ? $_SERVER['HTTP_X_WP_INTERNAL'] : '';
            if ($internal_header !== 'true') {
                // Missing header = potential CSRF or external request
                // Don't authenticate - user will be treated as guest
                $this->log_debug('authenticate_rest_from_cookie: Rejected - missing X-WP-Internal header');
                return;
            }
        }

        // Validate the cookie cryptographically (checks signature, expiration, etc.)
        $user_id = wp_validate_auth_cookie($_COOKIE[LOGGED_IN_COOKIE], 'logged_in');

        if ($user_id) {
            // Cookie valid - set current user for this request
            // This allows is_user_logged_in() and wp_get_current_user() to work
            wp_set_current_user($user_id);
            $this->log_debug("authenticate_rest_from_cookie: Authenticated user $user_id from cookie");
        }
        // If cookie invalid/expired: user remains guest (no error - silent fallback)
    }

    /**
     * Verify that the current session owns the requested cart.
     * Works for both logged-in users and guests.
     *
     * @param string $cart_id The cart ID from the request
     * @return object|WP_Error Returns cart row if owned, WP_Error if not
     */
    private function verify_cart_ownership($cart_id) {
        global $wpdb;

        // Validate cart_id format (10 hex chars - see generate_oa_basket_id())
        if (!preg_match('/^[a-f0-9]{10}$/i', $cart_id)) {
            return new WP_Error('invalid_cart_id', 'Invalid cart ID format', ['status' => 400]);
        }

        // Authenticate from cookies if not already logged in (REST API doesn't do this automatically)
        $this->authenticate_rest_from_cookie();

        // Ensure WC session is loaded (may not happen automatically in REST context)
        $this->ensure_wc_session_loaded();

        // Check if WooCommerce session is available
        if (!function_exists('WC') || !WC()->session) {
            return new WP_Error('no_session', 'Session not available', ['status' => 403]);
        }

        // Get current session ID and hash it (MD5 to match stored cart_session_id)
        $current_session_id = WC()->session->get_customer_id();

        if (empty($current_session_id)) {
            return new WP_Error('no_session', 'No active session', ['status' => 403]);
        }
        $current_session_hash = hash('md5', $current_session_id);

        // Fetch cart and verify ownership
        $cart_data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}oa_woocommerce_persistent_cart WHERE cart_id = %s AND cart_session_id = %s",
            $cart_id,
            $current_session_hash
        ));

        if (!$cart_data) {
            return new WP_Error('access_denied', 'Cart not found or access denied', ['status' => 403]);
        }

        return $cart_data;
    }

    /** @TODO */
    private function validate_discount($discount) {
        // Implement your validation logic here...
        // You could check if a coupon with the given code exists,
        // if it's still valid (not expired), etc.

        // For now, let's say all discounts are valid
        return true;
    }


    protected function encodeJsonResponse($response)
    {
        // return json_encode($response);
        return wp_json_encode($response, JSON_UNESCAPED_SLASHES);
    }


    private function create_product_output_array($_product, $quantity, $error = null) {
        $product_url = wp_get_attachment_url($_product->get_image_id());
        $images = $product_url ? array($product_url) : array();

        $unit_price = round($_product->get_price() * 100);
        $line_price = round($_product->get_price() * $quantity * 100);
        $originalUnitPrice = round($_product->get_price() * 100);
        $originalLinePrice = round($_product->get_price() * $quantity * 100);

        $product_data =  array(
            'ean' => $_product->get_sku(),
            'id' => (string)$_product->get_id(),
            'name' => $_product->get_name(),
            'images' => $images,
            'quantity' => $quantity,
            'unitPrice' => $unit_price,
            'linePrice' => $line_price,
            'originalUnitPrice' => $originalUnitPrice,
            'originalLinePrice' => $originalLinePrice
        );
        if($error){
            $product_data['error'] = $error;
        }

        return $product_data;

    }

    private function create_product_output_array_from_cart($item) {
        // Important! Return main product or variation
        $_product = $this->get_product_object($item);

        if(is_null($_product)){
            return array();
        }

        $product_url = wp_get_attachment_url($_product->get_image_id());
        $images = $product_url ? array($product_url) : array();

        // Unit prices before discount
        $original_unit_price_exclusive = $item['line_subtotal'] / $item['quantity'];
        $original_unit_tax = $item['line_subtotal_tax'] / $item['quantity'];
        $original_unit_price_inclusive = $original_unit_price_exclusive + $original_unit_tax;

        // Unit prices after discount
        $unit_price_exclusive = $item['line_total'] / $item['quantity'];
        $unit_tax = $item['line_tax'] / $item['quantity'];
        $unit_price_inclusive = $unit_price_exclusive + $unit_tax;

        // Line prices (for all quantities)
        $line_price_exclusive = $item['line_total'];
        $line_tax = $item['line_tax'];
        $line_price_inclusive = $line_price_exclusive + $line_tax;

        // Original line prices (for all quantities) before discount
        $original_line_price_exclusive = $item['line_subtotal'];
        $original_line_tax = $item['line_subtotal_tax'];
        $original_line_price_inclusive = $original_line_price_exclusive + $original_line_tax;

        $product_data = array(
            'ean' => $_product->get_sku(),
            'id' => (string)$_product->get_id(),
            'name' => $_product->get_name(),
            'images' => $images,
            'quantity' => $item['quantity'],
            'unitPrice' => round($unit_price_inclusive * 100),  // Converted to grosz
            'linePrice' => round($line_price_inclusive * 100),
            'originalUnitPrice' => round($original_unit_price_inclusive * 100),
            'originalLinePrice' => round($original_line_price_inclusive * 100),
        );

        return $product_data;
    }

    private function get_delivery_options() {
        $methods = $this->shipping_methods;

        $enabled_methods = array();

        foreach ($methods as $key => $method) {
            if ('yes' === get_option('woocommerce_openapp_' . $key . '_enabled')) {
                $enabled_methods[] = array(
                    'key' => $key,
                    // convert cost from decimal format to minor currency unit (cents)
                    'cost' => (int) (get_option('woocommerce_openapp_' . $key . '_cost') * 100)
                );
            }
        }

        return $enabled_methods;
    }


    private function get_discounts($coupons) {
        $discounts = array();

        foreach($coupons as $coupon) {
            $discounts[] = array(
                'code' => $coupon['code'],
                'value' => round($coupon['discount_amount'] * 100)
            );
        }

        return $discounts;
    }

    private function user_session_is_active($token){

        $woocommerceSessionActive = false;

        if (isset(WC()->session)) {
            if (WC()->session->has_session()) {
                $woocommerceSessionActive = true;
            }
        }

        if($woocommerceSessionActive && !is_null($token)){
            return true;
        }

        return false;
    }

    private function is_heartbeat() {
        $action = filter_input(INPUT_POST, 'action', FILTER_SANITIZE_SPECIAL_CHARS);
        return ($action == 'heartbeat');
    }

    private function base64Encode($data) {
        return base64_encode($data);
    }

    private function createWpUser($email, $random_password) {
        $user_id = wc_create_new_customer($email, '', $random_password);

        if (is_wp_error($user_id)) {
            // Handle the error.
            //echo 'Unable to create a new user: ' . $user_id->get_error_message();
            // return $user_id;
        }

        return $user_id;
    }

    /**
     * ===========================================================================
     * Testing / Debugging
     * ===========================================================================
     */

    public function log_debug($message) {
        if ('yes' === $this->get_option('debug', 'no')) {
            if (is_array($message)) {
                $message = print_r($message, true);
            }
            $timestamp = gmdate('Y-m-d H:i:s');
            error_log("[OpenApp {$timestamp}] " . $message);
        }
    }

    private function mask_secret($secret){
        $start = substr($secret, 0, 4); // Get the first four characters
        $end = substr($secret, -4); // Get the last four characters
        $masked = str_repeat('*', strlen($secret) - 8); // Generate a string of '*' with the same length as the remaining characters

        return $start . $masked . $end; // Return the masked secret
    }

}
