<?php
/**
 * HMAC Authentication trait for OpenApp Gateway.
 *
 * @package OpenApp_Payment_Gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Trait OPENAPPGW_Trait_HMAC_Security
 *
 * Handles HMAC signature generation and validation for API security.
 */
trait OPENAPPGW_Trait_HMAC_Security {

    private function hmacSignatureIsValid($authorizationHeader, $serverSignature, $responseBodyHashBase64 = null){
        // Split the HMAC header into its parts
        $components = explode('$', $authorizationHeader);

        // Validate header format: v1$api_key$method$path$timestamp$nonce
        if (count($components) < 6) {
            return false;
        }

        list($version, $api_key, $method, $path, $timestamp, $nonce) = $components;

        // Check if the timestamp is within the allowed 60 seconds window
        $currentTimestamp = round(microtime(true) * 1000); // Current time in milliseconds
        $diff = abs($currentTimestamp - $timestamp);

        if ($diff > 60000) {
            return false; // Reject if the timestamp difference is greater than 60 seconds
        }

        $hmacSignature = $this->generateHmacSignature($method, $path, $timestamp, $nonce, $responseBodyHashBase64, $api_key, $this->secret);

        if(hash_equals($serverSignature, $hmacSignature)){
            return true;
        } else {
            return false;
        }
    }

    protected function generateHmacSignature($method, $path, $timestamp, $nonce, $content = null, $api_key = null, $secret = null, $log_to_context = '') {
        if(!$api_key){
            $api_key = $this->api_key;
        }
        if(!$secret){
            $secret = $this->secret;
        }

        $stringToSign = "v1$"."$api_key$" . "$method$" . "$path$" . "$timestamp$" . "$nonce";

        if ($content) {
            $stringToSign .= "$".$content;
        }

        if($log_to_context){
            $this->openappgw_custom_log("STRING_TO_SIGN: " . $stringToSign, $log_to_context);
        }


        $hmacHash = hash_hmac('sha256', $stringToSign, $secret, true);
        $hmacSignature = $this->base64Encode($hmacHash);

        return $hmacSignature;
    }


    private function calculate_server_authorization($headers, $responseBody = null) {
        $authorization = isset($headers["authorization"]) ? $headers["authorization"][0] : null;

        if(!is_null($authorization)) {
            $components = explode('$', $authorization);

            if (count($components) < 6) {
                // The authorization string doesn't contain as many components as we're expecting.
                return null;
            }


            $timestamp = $components[4];
            $nonce = $components[5];

            $calculatedSignature = $this->generateHmacResponseSignature($timestamp, $nonce, $responseBody, $authorization);

            $context = 'openapp_hmac_response';
            $this->openappgw_custom_log("calculatedSignature: ". $calculatedSignature, $context);

            return "hmac v1". "$"."$timestamp"."$".$nonce."$"."$calculatedSignature";
        }

        return null;
    }

    protected function calculate_x_server_auth($headers, $responseBody = null) {
        $authorization = isset($headers["x-server-authorization"]) ? $headers["x-server-authorization"] : null;

        if(!is_null($authorization)) {
            $components = explode('$', $authorization);

            if (count($components) < 4) {
                // The authorization string doesn't contain as many components as we're expecting.
                return null;
            }

            $timestamp = $components[1];
            $nonce = $components[2];

            $calculatedSignature = $this->generateHmacResponseSignature($timestamp, $nonce, $responseBody, $authorization);

            $context = 'openapp_hmac_response';
            $this->openappgw_custom_log("calculatedSignature X-Server-Auth: ". $calculatedSignature, $context);

            $hmacXServerHeader = "hmac v1". "$"."$timestamp"."$".$nonce."$"."$calculatedSignature";

            return $hmacXServerHeader;
        }

        return null;
    }


    protected function generateHmacResponseSignature($timestamp,$nonce,$responseBody = null, $authorization = null, $printDebug = false) {

        $stringToSign =  "v1$"."$timestamp"."$"."$nonce";

        if (!is_null($responseBody)) {
            // FIX (encode body to convert backslash characters)
            $responseBody = wp_json_encode($responseBody);

            $responseBodyHash = hash('sha256', $responseBody, true);

            $responseBodyHashBase64 = $this->base64Encode($responseBodyHash);
            $stringToSign .= "$" . $responseBodyHashBase64;
        }


        $context = 'openapp_hmac_response';
        $this->openappgw_custom_log("GENERATE_HMAC_RESPONSE_SIGNATURE", $context);
        $this->openappgw_custom_log("Request Authorization Header: ". $authorization, $context);
        $this->openappgw_custom_log("Timestamp: ".$timestamp, $context);
        $this->openappgw_custom_log("Nonce: ".$nonce, $context);
        $this->openappgw_custom_log("responseBody: ". var_export($responseBody, true), $context);
        if(isset($responseBodyHash)){
            $this->openappgw_custom_log("responseHash: ".$responseBodyHash, $context);
        }
        if(isset($responseBodyHashBase64)){
            $this->openappgw_custom_log("responseBodyHashBase64: ".$responseBodyHashBase64, $context);
        }

        $this->openappgw_custom_log("stringToSign: ".$stringToSign, $context);
        $this->openappgw_custom_log("-----------------------", $context);

        $hmacHash = hash_hmac('sha256', $stringToSign, $this->secret, true);
        $hmacHashBase64 = $this->base64Encode($hmacHash);

        if($printDebug){
            // Output
            var_dump("GENERATING X-Server-Authorization Header");
            var_dump("Secret: ". $this->mask_secret($this->secret));
            var_dump("bodyDigest: " . $responseBodyHashBase64);
            var_dump("Nonce: " . $nonce);
            var_dump("Signature: " . $hmacHashBase64 );
            var_dump("Timestamp: " . $timestamp);
            var_dump("StringToSign: " . $stringToSign);
        }

        return $hmacHashBase64;
    }



    private function isRequestValid($headers, $body = null) {
        if (!isset($headers['authorization'][0]) || !isset($headers['x_app_signature'][0])) {
            return false;
        }

        // Check the value of the validation_enabled option
        if ('no' === $this->get_option('validation_enabled', 'yes')) {
            return true;
        }

        $validHmac = $this->hmacSignatureIsValid($headers['authorization'][0], $headers['x_app_signature'][0], $body);

        $context = 'openapp_hmac_response';
        $this->openappgw_custom_log("isRequestValid Auth: ". var_export($headers['authorization'][0], true), $context);
        $this->openappgw_custom_log("isRequestValid X-App-Signature: ". var_export($headers['x_app_signature'][0], true), $context);
        $this->openappgw_custom_log("isRequestValid Body: ". var_export($body, true), $context);
        $this->openappgw_custom_log("isRequestValid: ". var_export($validHmac, true), $context);
        $this->openappgw_custom_log("---------------------", $context);

        return $validHmac;
    }

}
