<?php
/**
 * REST API Handlers trait for OpenApp Gateway.
 *
 * @package OpenApp_Payment_Gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Trait OPENAPPGW_Trait_REST_API_Handlers
 *
 * Handles REST API endpoint implementations for products, basket, and orders.
 */
trait OPENAPPGW_Trait_REST_API_Handlers {

    public function retrieve_products(WP_REST_Request $request) {
        try {
            $full_data = $request->get_param('full');
            $headers = $request->get_headers();

            // Pagination parameters - only use if explicitly provided
            $page = $request->get_param('page');
            $per_page = $request->get_param('per_page');
            $use_pagination = !empty($page) || !empty($per_page);

            if ($use_pagination) {
                $page = (int) $page ?: 1;
                $per_page = (int) $per_page ?: 50;
            }

            // Logger
            $context = 'openapp_retrieve_products';
            $this->openappgw_custom_log("Full data: " . print_r($full_data, true), $context);
            $this->openappgw_custom_log("Pagination: page={$page}, per_page={$per_page}, use_pagination=" . ($use_pagination ? 'yes' : 'no'), $context);
            $this->openappgw_custom_log("Headers: " . print_r($this->encodeJsonResponse($headers), true), $context);

            // Verify HMAC
            $validHmac = $this->isRequestValid($headers);
            if (!$validHmac) {
                return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
            }

            // Fetch products efficiently
            $args = array(
                'limit' => -1,
                'type' => ['variable', 'simple']
            );
            $products = wc_get_products($args);
        } catch (Exception $e) {
            $this->openappgw_custom_log("Error in retrieve_products: " . $e->getMessage(), 'openapp_retrieve_products');
            return new WP_Error('product_retrieval_error', 'Failed to retrieve products: ' . $e->getMessage(), array('status' => 500));
        }
        try {
            $response = array();
            $current_count = 0;
            $items_added = 0;
            $offset = $use_pagination ? ($page - 1) * $per_page : 0;

            // Get units once for performance
            $weight_unit = get_option('woocommerce_weight_unit');
            $dimension_unit = get_option('woocommerce_dimension_unit');

            foreach ($products as $product) {
                if ($use_pagination && $items_added >= $per_page) {
                    break;
                }

                try {
                    // Check if the product is a variable product
                    if ($product->is_type('variable')) {
                        // Get the variations
                        $variations = $product->get_children();

                        foreach ($variations as $variation_id) {
                            if ($use_pagination && $current_count < $offset) {
                                $current_count++;
                                continue;
                            }

                            if ($use_pagination && $items_added >= $per_page) {
                                break;
                            }

                            try {
                                $_variation = wc_get_product($variation_id);
                                if ($_variation) {
                                    $response[] = $this->get_product_response($_variation, $full_data === 'yes', $weight_unit, $dimension_unit);
                                    $current_count++;
                                    if ($use_pagination) $items_added++;
                                }
                            } catch (Exception $e) {
                                $this->openappgw_custom_log("Error processing variation {$variation_id}: " . $e->getMessage(), $context);
                                // Continue processing other variations
                            }
                        }
                    } else {
                        if ($use_pagination && $current_count < $offset) {
                            $current_count++;
                            continue;
                        }

                        // For simple products
                        $response[] = $this->get_product_response($product, $full_data === 'yes', $weight_unit, $dimension_unit);
                        $current_count++;
                        if ($use_pagination) $items_added++;
                    }
                } catch (Exception $e) {
                    $product_id = $product ? $product->get_id() : 'unknown';
                    $this->openappgw_custom_log("Error processing product {$product_id}: " . $e->getMessage(), $context);
                    // Continue processing other products
                }
            }
        } catch (Exception $e) {
            $this->openappgw_custom_log("Error in product loop: " . $e->getMessage(), $context);
            return new WP_Error('product_processing_error', 'Failed to process products: ' . $e->getMessage(), array('status' => 500));
        }

        // Return response with or without pagination metadata
        if ($use_pagination) {
            // Calculate accurate total items for pagination
            $total_items = 0;
            foreach ($products as $product) {
                if ($product->is_type('variable')) {
                    $total_items += count($product->get_children());
                } else {
                    $total_items++;
                }
            }

            $total_pages = ceil($total_items / $per_page);

            $paginated_response = array(
                'data' => $response,
                'pagination' => array(
                    'page' => $page,
                    'per_page' => $per_page,
                    'total_items' => $total_items,
                    'total_pages' => $total_pages,
                    'has_next_page' => $page < $total_pages,
                    'has_prev_page' => $page > 1
                )
            );

            // Create WP_REST_Response object
            $wpRestResponse = new WP_REST_Response($paginated_response, 200);

            // Calculate X-Server-Authorization and set the header if it exists
            $expectedXServerAuth = $this->calculate_server_authorization($headers, $paginated_response);
            if ($expectedXServerAuth !== null) {
                $this->openappgw_custom_log("X-Server-Authorization: " . $expectedXServerAuth, $context);
                $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
            }

            return $wpRestResponse;
        } else {
            // Return all items without pagination wrapper

            // Create WP_REST_Response object
            $wpRestResponse = new WP_REST_Response($response, 200);

            // Calculate X-Server-Authorization and set the header if it exists
            $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
            if ($expectedXServerAuth !== null) {
                $this->openappgw_custom_log("X-Server-Authorization: " . $expectedXServerAuth, $context);
                $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
            }

            return $wpRestResponse;
        }
    }

    // Helper function to format product/variation data
    private function get_product_response($_product, $full, $weight_unit = '', $dimension_unit = '') {
        // Get product images
        $images = array();
        foreach ($_product->get_gallery_image_ids() as $image_id) {
            $image_url = wp_get_attachment_image_url($image_id, 'full');
            if ($image_url) {
                $images[] = $image_url;
            }
        }

        // Get main product image
        $main_image = wp_get_attachment_image_url($_product->get_image_id(), 'full');
        if ($main_image) {
            array_unshift($images, $main_image); // Make sure main image is first
        }

        // Prepare stock data
        // important - to have param edit
        $manage_stock = $_product->get_manage_stock('edit');
        $stock_quantity = $manage_stock ? $_product->get_stock_quantity() : null;

        // Get categories and tags (for full data only)
        // Determine if the product is a variation and get its parent
        $parent_id = $_product->is_type('variation') ? $_product->get_parent_id() : $_product->get_id();
        $parent_product = wc_get_product($parent_id);

        $categories = array();
        $tags = array();

        if ($full) {
            // Fetch category names from parent product
            $category_ids = $parent_product->get_category_ids();
            foreach ($category_ids as $category_id) {
                $category = get_term($category_id, 'product_cat');
                if (!is_wp_error($category)) {
                    $categories[] = $category->name;
                }
            }

            // Fetch tag names from parent product
            $tag_ids = $parent_product->get_tag_ids();
            foreach ($tag_ids as $tag_id) {
                $tag = get_term($tag_id, 'product_tag');
                if (!is_wp_error($tag)) {
                    $tags[] = $tag->name;
                }
            }
        }

        // Get description and short_description, fallback to parent if variation is empty
        $description = $_product->get_description();
        if (empty($description) && $_product->is_type('variation')) {
            $description = $parent_product->get_description();
        }

        $short_description = $_product->get_short_description();
        if (empty($short_description) && $_product->is_type('variation')) {
            $short_description = $parent_product->get_short_description();
        }

        if ($full) {
            return array(
                'id' => (string) $_product->get_id(),
                'sku' => $_product->get_sku(),
                'ean' => $_product->get_global_unique_id(),
                'name' => $_product->get_name(),
                'type' => $_product->get_type(),
                'description' => $description,
                'short_description' => $short_description,
                'price' => $this->convertToCents($_product->get_price()),
                'regular_price' => $this->convertToCents($_product->get_regular_price()),
                'sale_price' => $this->convertToCents($_product->get_sale_price()),
                'stock_status' => $_product->get_stock_status(),
                'stock_quantity' => $stock_quantity,
                'weight' => $_product->get_weight(),
                'weight_unit' => $weight_unit,
                'length' => $_product->get_length(),
                'width' => $_product->get_width(),
                'height' => $_product->get_height(),
                'dimension_unit' => $dimension_unit,
                'permalink' => $_product->get_permalink(),
                'images' => $images,
                'categories' => $categories,
                'tags' => $tags,
                'attributes' => $_product->is_type('variation') ? $_product->get_attributes() : null, // Variation attributes
            );
        } else {
            return array(
                'id' => (string) $_product->get_id(),
                'sku' => $_product->get_sku(),
                'ean' => $_product->get_global_unique_id(),
                'name' => $_product->get_name(),
                'price' => $this->convertToCents($_product->get_price()),
                'regular_price' => $this->convertToCents($_product->get_regular_price()),
                'sale_price' => $this->convertToCents($_product->get_sale_price()),
                'stock_status' => $_product->get_stock_status(),
                'stock_quantity' => $stock_quantity
            );
        }
    }

    public function retrieve_basket( WP_REST_Request $request) {
        // Fetch the basketId from the request
        $basketId = $request->get_param('basketId');

        // Fetch the headers from the request
        $headers = $request->get_headers();

        // logger
        $context = 'openapp_retrieve_basket';
        $this->openappgw_custom_log("Basket ID: ".print_r($basketId, true), $context);
        $this->openappgw_custom_log("Headers: ".print_r($this->encodeJsonResponse($headers), true), $context); // Logs the headers

        // Verify HMAC
        $validHmac = $this->isRequestValid($headers);

        if(!$validHmac){
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        // Retrieve the basket data from your database
        $response = $this->get_basket_data($basketId);

        if (empty($response)) {
            return new WP_Error('basket_not_found', 'Basket not found', array('status' => 404));
        }


        // Create WP_REST_Response object
        $wpRestResponse = new WP_REST_Response($response, 200);

        $this->openappgw_custom_log("responseBody: ". var_export($response, true), $context);

        // Calculate X-Server-Authorization and set the header if it exists
        $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);

        if($expectedXServerAuth !== null) {
            $this->openappgw_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }

        $this->openappgw_custom_log("------------------------------------", $context);

        return $wpRestResponse;
    }

    public function create_or_update_basket(WP_REST_Request $request) {
        /**
         * Do not store cart in DB
         */
        remove_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        $headers = $request->get_headers();

        // Validate HMAC
        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if (!$validHmac) {
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        $data = $request->get_json_params();

        // Validate input model
        if (empty($data['basket']) || empty($data['basket']['products'])) {
            return new WP_Error('invalid_data', 'basket and products are required', array('status' => 400));
        }

        $basket = $data['basket'];
        $products = $basket['products'];
        $loggedUser = $basket['loggedUser'] ?? null;

        $currency = get_woocommerce_currency();
        $cart_id = $basket['id'] ?? $this->get_unique_oa_basket_id();

        // --- Reuse your recalc/init logic ---
        $output = $this->initialize_output($cart_id, $currency, $loggedUser);

        // Get products with adjusted quantities and errors
        $adjusted_products = $this->get_stock_adjusted_products($products);

        // Extract errors by product ID for merging later
        $stock_errors = [];
        foreach ($adjusted_products as $p) {
            if (isset($p['error'])) {
                $stock_errors[$p['id']] = $p['error'];
            }
        }

        $cart_contents_record = $this->build_cart_contents_record([
            'products' => $adjusted_products,
            'price' => [ 'discounts' => [] ], // Adjust if you want to support input discounts
            'loggedUser' => $loggedUser
        ]);

        // Build products from discounted cart contents with stock errors
        $output['products'] = $this->get_products_from_cart_contents_with_errors(
            $cart_contents_record['cart_contents'],
            $stock_errors
        );
        $output['price']['basketValue'] = $cart_contents_record['basketValue'];
        $output['price']['discounts'] = $cart_contents_record['applied_discounts'];

        $output['deliveryOptions'] = $this->get_available_shipping_methods(
            $this->supported_country,
            $cart_contents_record['basketValue'],
            $cart_contents_record
        );

        $wpRestResponse = new WP_REST_Response($output, 200);

        $expectedXServerAuth = $this->calculate_server_authorization($headers, $output);
        if ($expectedXServerAuth !== null) {
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }

        // Re-add the action after calculations are done
        // not sure, if it is needed..
        add_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        return $wpRestResponse;
    }




    public function basket_recalculate(WP_REST_Request $request) {
        /**
         * Do not store cart in DB
         */
        remove_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        $data = $request->get_json_params();
        $headers = $request->get_headers();

        $context = 'openapp_basket_recalculate';
        $this->openappgw_custom_log(print_r($data, true), $context);
        $this->openappgw_custom_log(print_r($this->get_values_and_types($data), true), $context);
        $this->openappgw_custom_log("Headers: ".print_r($this->encodeJsonResponse($headers), true), $context);

        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if(!$validHmac){
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        $currency = get_woocommerce_currency();
        $output = $this->initialize_output($data['id'], $currency, $data['loggedUser'] ?? null);

        // Get products with adjusted quantities and errors
        $adjusted_products = $this->get_stock_adjusted_products($data['products']);

        // Extract errors by product ID for merging later
        $stock_errors = [];
        foreach ($adjusted_products as $p) {
            if (isset($p['error'])) {
                $stock_errors[$p['id']] = $p['error'];
            }
        }

        // Build cart_contents_record with adjusted quantities (applies discounts)
        $cart_contents_record = $this->build_cart_contents_record([
            'products' => $adjusted_products,
            'price' => $data['price'],
            'loggedUser' => $data['loggedUser']
        ]);

        // Build products from discounted cart contents with stock errors
        $output['products'] = $this->get_products_from_cart_contents_with_errors(
            $cart_contents_record['cart_contents'],
            $stock_errors
        );
        $output['price']['basketValue'] =  $cart_contents_record['basketValue'];
        $output['price']['discounts'] =  $cart_contents_record['applied_discounts'];


        $output['deliveryOptions'] = $this->get_available_shipping_methods(
            $this->supported_country,
            $cart_contents_record['basketValue'],
            $cart_contents_record
        );


        $this->openappgw_custom_log("RESPONSE:", $context);
        $this->openappgw_custom_log(print_r($output, true), $context);
        $this->openappgw_custom_log("------------------------------------", $context);

        $wpRestResponse = new WP_REST_Response($output, 200);

        $expectedXServerAuth = $this->calculate_server_authorization($headers, $output);
        if ($expectedXServerAuth !== null) {
            $this->openappgw_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }


        // Re-add the action after calculations are done
        // not sure, if it is needed..
        add_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        return $wpRestResponse;
    }


    private function build_cart_contents_record($data)
    {
        // Create a temporary WC_Cart instance
        $fake_cart = new WC_Cart();

        // Add products to the fake cart
        foreach ($data['products'] as $product_data) {
            $product_id = $product_data['id'];
            $quantity = $product_data['quantity'];
            $variation_id = $product_data['variation_id'] ?? 0;
            $variation = $product_data['variation'] ?? [];

            // Add product to the fake cart
            $fake_cart->add_to_cart($product_id, $quantity, $variation_id, $variation);
        }

        // Calculate totals before validating coupons (required for minimum amount checks)
        $fake_cart->calculate_totals();

        $discounts = new WC_Discounts($fake_cart);

        // Apply each discount code from the data
        foreach ($data['price']['discounts'] as $discount) {
            $coupon_code = $discount['code'];
            $coupon = new WC_Coupon($coupon_code);

            // Use WC_Discounts->is_coupon_valid to check if the coupon is valid
            $is_valid = $discounts->is_coupon_valid($coupon);
            // is_coupon_valid returns true on success or WP_Error on failure
            if ($is_valid === true || !is_wp_error($is_valid)) {
                // Add coupon directly to applied_coupons array instead of using apply_coupon
                // which requires session context that doesn't exist for fake carts
                $fake_cart->applied_coupons[] = strtolower($coupon_code);
            }
        }

        // Calculate totals to ensure discounts are applied
        $fake_cart->calculate_totals();

        // Retrieve cart_contents from the fake cart
        $cart_contents = $fake_cart->get_cart();

        // Prepare cart_contents_data with additional product info
        $cart_contents_data = [];
        foreach ($cart_contents as $key => $item) {
            $wc_product = wc_get_product($item['product_id']);
            if ($wc_product) {
                $cart_contents_data[$key] = [
                    'name' => $wc_product->get_name(),
                    'shippingClass' => $wc_product->get_shipping_class(),
                    'weight' => $wc_product->get_weight(),
                    'needsShipping' => $wc_product->needs_shipping()
                ];
            }
        }

        $applied_discounts = [];
        foreach ($fake_cart->get_applied_coupons() as $coupon_code) {
            $discount_amount = $fake_cart->get_coupon_discount_amount($coupon_code);

            $applied_discounts[] = [
                'code' => $coupon_code,
                'value' => $this->convertToCents($discount_amount)
            ];
        }

        $basketValue = $this->calculate_total_price($cart_contents);

        return [
            'cart_contents' => $cart_contents,
            'coupon_data' => $data['price']['discounts'],
            'user_id' => $data['loggedUser'] ?? null,
            'cart_contents_data' => $cart_contents_data,
            'basketValue' => $basketValue,
            'applied_discounts' => $applied_discounts
        ];
    }


    private function adjust_quantities($products) {
        $adjusted_products = [];

        foreach ($products as $product_data) {
            $wc_product = wc_get_product($product_data['id']);
            if (!$wc_product) continue;

            $quantity = $product_data['quantity'];
            $error = $this->check_stock_availability($wc_product, $quantity);

            // $quantity is already adjusted within check_stock_availability if needed
            $adjusted_products[] = $this->create_product_output_array($wc_product, $quantity, $error);
        }

        return $adjusted_products;
    }

    private function initialize_output($id, $currency, $loggedUser = null) {

        $cart_id = $id ?? $this->get_unique_oa_basket_id();

        $output = [
            'id' => $cart_id,
            'expiresAt' => gmdate('Y-m-d\TH:i:s\Z', strtotime('+1 day')),
            'price' => [
                'currency' => $currency,
                'discounts' => [],
                'basketValue' => 0
            ],
            'deliveryOptions' => [],
            'products' => []
        ];

        if ($loggedUser) {
            $output['loggedUser'] = $loggedUser;
        }

        return $output;
    }

    private function check_stock_availability($wc_product, &$quantity) {
        $error = null;

        // Check if the product is in stock
        if (!$wc_product->is_in_stock()) {
            $error = "OUT_OF_STOCK";
            $quantity = 0;
        } elseif ($wc_product->managing_stock()) {
            // If managing stock, check the stock quantity
            $stock_quantity = $wc_product->get_stock_quantity();
            if ($quantity > $stock_quantity) {
                $quantity = $stock_quantity;
                $error = "QUANTITY_TOO_BIG";
            }
        }

        return $error;
    }

    private function get_stock_adjusted_products($products) {
        $adjusted = [];
        foreach ($products as $product_data) {
            // For variations, check stock on variation_id, not parent id
            // Note: Use !empty() instead of ?? because variation_id=0 is falsy but not null
            $product_id = !empty($product_data['variation_id']) ? $product_data['variation_id'] : $product_data['id'];
            $wc_product = wc_get_product($product_id);
            if (!$wc_product) continue;

            $quantity = $product_data['quantity'];
            $error = $this->check_stock_availability($wc_product, $quantity);

            // Build adjusted product with corrected quantity and error
            $adjusted_product = $product_data;
            $adjusted_product['quantity'] = $quantity;  // Now adjusted by check_stock_availability
            if ($error) {
                $adjusted_product['error'] = $error;
            }
            $adjusted[] = $adjusted_product;
        }
        return $adjusted;
    }

    private function get_products_from_cart_contents_with_errors($cart_contents, $stock_errors = []) {
        $products = [];
        foreach ($cart_contents as $item) {
            $product_data = $this->create_product_output_array_from_cart($item);
            if (!empty($product_data)) {
                $product_id = $product_data['id'];
                if (isset($stock_errors[$product_id])) {
                    $product_data['error'] = $stock_errors[$product_id];
                }
                $products[] = $product_data;
            }
        }
        return $products;
    }


    /**
    * Place order
     */
    public function create_new_wc_order(WP_REST_Request $request) {
        // Parse and validate request body
        $data = $request->get_json_params();

        // Fetch the headers from the request
        $headers = $request->get_headers();

        // logger
        $context = 'openapp_place_order';
        $this->openappgw_custom_log(print_r($data, true), $context);
        $this->openappgw_custom_log(print_r($this->get_values_and_types($data), true), $context);
        $this->openappgw_custom_log("Headers: ".print_r($this->encodeJsonResponse($headers), true), $context); // Logs the headers

        if (empty($data['oaOrderId'])) {
            return new WP_Error('missing_oaOrderId', 'OA Order ID is required', array('status' => 400));
        }

        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if(!$validHmac){
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        $cart_id = $data['basket']['id'];

        // Validate all products exist before creating the order
        $invalid_products = array();
        foreach ($data['basket']['products'] as $product) {
            $product_to_add = wc_get_product($product['id']);
            if (!$product_to_add) {
                $invalid_products[] = $product['id'];
            }
        }

        if (!empty($invalid_products)) {
            return new WP_REST_Response( array(
                'message' => 'Error: Product(s) with id(s) ' . implode(', ', $invalid_products) . ' do not exist',
            ), 400 );
        }

        // All products valid, now create order
        $order = wc_create_order();

        foreach ($data['basket']['products'] as $product) {
            $product_to_add = wc_get_product($product['id']);
            $order->add_product($product_to_add, $product['quantity']);
        }


        // Here, we apply the discounts
        if (!empty($data['basket']['price']['discounts'])) {
            foreach ($data['basket']['price']['discounts'] as $discount) {
                // You could implement a function that validates the discount before applying it
                if ($this->validate_discount($discount)) {
                    $coupon = new WC_Coupon($discount['code']);

                    // Create an instance of the WC_Discounts class.
                    $discounts = new WC_Discounts( $order );

                    // Check if the coupon is valid for this order.
                    $valid = $discounts->is_coupon_valid( $coupon );

                    if ( is_wp_error( $valid ) ) {
                        // The coupon is not valid. You can log the error message if you wish
                        // error_log( $valid->get_error_message() );
                    } else {
                        $order->apply_coupon($coupon->get_code());
                    }
                }
            }
        }

        /**
         * Save data start
         */
        // Update order details
        if (!empty($data['billingDetails'])) {
            $order->set_billing_company($data['billingDetails']['companyName'] ?? '');
            $order->set_billing_first_name($data['billingDetails']['firstName'] ?? '');
            $order->set_billing_last_name($data['billingDetails']['lastName'] ?? '');

            $billing_address_1 = ($data['billingDetails']['street'] ?? '') . ' ' . ($data['billingDetails']['streetNo'] ?? '');
            if(!empty($data['billingDetails']['apartmentNo'])){
                $billing_address_1 .= ' / ' . $data['billingDetails']['apartmentNo'];
            }
            $order->set_billing_address_1($billing_address_1);
            $order->set_billing_city($data['billingDetails']['city'] ?? '');
            $order->set_billing_postcode($data['billingDetails']['postalCode'] ?? '');
            $order->set_billing_country($data['billingDetails']['country'] ?? '');
            $order->set_billing_phone($data['billingDetails']['phoneNumber'] ?? '');

            if(!empty($data['billingDetails']['notes'])){
                $order->set_customer_note($data['billingDetails']['notes']);
            }
        }


        // Save email
        $order->set_billing_email($data['deliveryDetails']['email'] ?? '');

        // Set payment method
        $order->set_payment_method($this->id);
        $order->set_payment_method_title($this->payment_method_title);

        // Update shipping details
        // Concatenate street number and apartment number
        $address_line_1 = ($data['deliveryDetails']['street'] ?? '') . ' ' . ($data['deliveryDetails']['streetNo'] ?? '');
        if(!empty($data['deliveryDetails']['apartmentNo'])){
            $address_line_1 .= ' / ' . $data['deliveryDetails']['apartmentNo'];
        }
        $order->set_shipping_address_1($address_line_1);

        // $order->set_shipping_address_2('');
        $order->set_shipping_company($data['deliveryDetails']['companyName'] ?? '');
        $order->set_shipping_first_name($data['deliveryDetails']['firstName'] ?? '');
        $order->set_shipping_last_name($data['deliveryDetails']['lastName'] ?? '');
        $order->set_shipping_phone($data['deliveryDetails']['phoneNumber'] ?? '');
        $order->set_shipping_city($data['deliveryDetails']['city'] ?? '');
        $order->set_shipping_postcode($data['deliveryDetails']['postalCode'] ?? '');
        $order->set_shipping_country($data['deliveryDetails']['country'] ?? '');

        // Paczkomaty - Handle all APM (Automated Parcel Machine) delivery types
        $apm_methods = array('INPOST_APM', 'ORLEN_APM', 'POCZTA_POLSKA_APM');
        $delivery_method = isset($data['deliveryDetails']['method']) ? $data['deliveryDetails']['method'] : '';
        $delivery_subtype = isset($data['deliveryDetails']['subType']) ? $data['deliveryDetails']['subType'] : '';

        // Check if this is an APM delivery (either by method or subType)
        if (in_array($delivery_method, $apm_methods) || $delivery_subtype === 'APM') {
            if(isset($data['deliveryDetails']['id'])) {
                $parcelLockerId = $data['deliveryDetails']['id'];
                // Set the paczkomat ID as shipping company for all APM types
                $order->set_shipping_company($parcelLockerId);

                // InPost-specific meta fields for third-party plugin compatibility
                if ($delivery_method === 'INPOST_APM') {
                    // Save the Parcel Locker ID
                    update_post_meta($order->get_id(), 'paczkomat_key', $parcelLockerId);

                    // Construct and save the full name and address of the parcel locker
                    $parcelLockerFull = $data['deliveryDetails']['id'] . ', ' .
                        $data['deliveryDetails']['street'] . ' ' .
                        $data['deliveryDetails']['streetNo'] . ', ' .
                        $data['deliveryDetails']['postalCode'] . ' ' .
                        $data['deliveryDetails']['city'];
                    update_post_meta($order->get_id(), 'Wybrany paczkomat', $parcelLockerFull);
                }
            }
        }


        // save Notes + TaxID
        $taxIdNote = "";
        if(!empty($data['billingDetails']['taxId'])){
            $taxId = $data['billingDetails']['taxId'];
            $taxIdNote = "TaxId: ".$taxId . "\n";
            // Save NIP for Baselinker
            update_post_meta($order->get_id(), '_billing_nip', $taxId);
        }

        $deliveryNotes = isset($data['deliveryDetails']['notes']) ? $data['deliveryDetails']['notes'] : '';
        $billingNotes = isset($data['billingDetails']['notes']) ? $data['billingDetails']['notes'] : '';

        $order->add_order_note($taxIdNote . $deliveryNotes . "\n" . $billingNotes);

        // save note2
        $note2 = "OA Order ID: " . $data['oaOrderId'] . "\n";
        $note2 .= "Payment Currency: " . $data['paymentDetails']['currency'] . "\n";
        $note2 .= "Payment Amount: " . $data['paymentDetails']['amount'] . "\n";
        $note2 .= "Delivery Type: " . ($data['deliveryDetails']['type'] ?? 'N/A');

        // Add delivery method if available
        if (!empty($data['deliveryDetails']['method'])) {
            $note2 .= "\nDelivery Method: " . $data['deliveryDetails']['method'];
        }

        // Add delivery subType if available
        if (!empty($data['deliveryDetails']['subType'])) {
            $note2 .= "\nDelivery SubType: " . $data['deliveryDetails']['subType'];
        }

        // Add paczkomat/pickup point ID if available
        if (!empty($data['deliveryDetails']['id'])) {
            $note2 .= "\nPickup Point ID: " . $data['deliveryDetails']['id'];
        }

        $order->add_order_note($note2);

        // Update orderOAID
        update_post_meta($order->get_id(), 'oaOrderId', $data['oaOrderId']);


        /**
         * Save data end
         */

        // Delivery method
        $deliveryMethodKey = isset($data['deliveryDetails']['method']) ? $data['deliveryDetails']['method'] : null;
        $deliveryMethodName = isset($this->shipping_methods[$deliveryMethodKey]) ? $this->shipping_methods[$deliveryMethodKey] : $deliveryMethodKey;

        // Add the shipping method to the WooCommerce order
        $deliveryCost = isset($data['basket']['price']['deliveryCost']) ? $data['basket']['price']['deliveryCost'] : 0;
        $deliveryCost = $deliveryCost / 100; // converts groszy to zloty (adjust as necessary)

        // Baselinker - shipping method Paczkomaty
        $shippingItem = new WC_Order_Item_Shipping();
        $shippingItem->set_method_title($deliveryMethodName);
        $shippingItem->set_method_id($deliveryMethodKey);
        $shippingItem->set_total( $deliveryCost);

        $calculate_tax_for = array(
            'country' => $this->supported_country,
        );

        $taxesAmount = 0;

        if (wc_tax_enabled()) {
            $tax_rates = WC_Tax::find_shipping_rates($calculate_tax_for);
            $taxes = WC_Tax::calc_tax($deliveryCost, $tax_rates, true);
            $taxesAmount = array_sum($taxes); // Calculating sums of all taxes
        }

        $netDeliveryCost = $deliveryCost - $taxesAmount;
        $shippingItem->set_total($netDeliveryCost);
        $order->add_item($shippingItem);

        $order->calculate_totals();

        $order_id = $order->save();

        if($order_id){
            $max_return_days = 30;

            $response = array(
                "shopOrderId" => (string)$order_id,
                "returnPolicy" => array(
                    "maxReturnDays" => $max_return_days,
                )
            );

            // Increment order_count for the basket in your custom table
            global $wpdb;

            $wpdb->query(
                $wpdb->prepare(
                    "UPDATE " . $wpdb->prefix . "oa_woocommerce_persistent_cart SET order_count = order_count + 1, order_key = %s, oaOrderId = %s WHERE cart_id = %s",
                    $order_id,
                    $data['oaOrderId'],
                    $cart_id
                )
            );

            // Trigger payment complete
            $txn_id = $data['oaOrderId'] . '_' . date('ymd_His');
            $order->payment_complete($txn_id);

            /**
             * Once all processing is complete, update the order status
             */
            // Fetch the selected order status from the plugin settings
            $selected_status = $this->get_option('order_status', 'wc-processing');
            $selected_status_val = str_replace('wc-', '', $selected_status);

            // Update order status
            $order->update_status($selected_status_val);

            // Create WP_REST_Response object
            $wpRestResponse = new WP_REST_Response( $response, 200 );


            // Calculate X-Server-Authorization and set the header if it exists
            $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
            // Set X-Server-Authorization header if it exists
            if($expectedXServerAuth !== null) {
                $this->openappgw_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
                $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
            }

            $this->openappgw_custom_log("------------------------------------", $context);

            return $wpRestResponse;

        }else{
            return new WP_REST_Response( 'Error in Order Creation', 500 );
        }
    }

    public function handle_identity( WP_REST_Request $request ) {
        // Parse and validate request body
        $data = $request->get_json_params();

        // Fetch the headers from the request
        $headers = $request->get_headers();

        // logger
        $context = 'openapp_identity';
        $this->openappgw_custom_log(print_r($data, true), $context);
        $this->openappgw_custom_log(print_r($this->get_values_and_types($data), true), $context);
        $this->openappgw_custom_log("Headers: ".print_r($this->encodeJsonResponse($headers), true), $context); // Logs the headers

        $this->openappgw_custom_log("------------------------------------", $context);

        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if(!$validHmac){
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        // Sanitize and get the parameters
        $email = sanitize_email($data[ 'email' ]);
        $token = sanitize_text_field($data[ 'token' ]);


        if(empty($email) || empty($token)){
            return new WP_Error('missing_email_token', 'Incorrect parameters', array('status' => 400));
        }
        // check if token (cart_id) exists
        global $wpdb;

        $cart_data = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $token)
        );

        if (is_null($cart_data)) {
            return new WP_REST_Response( "Session do not exists", 500 );
        }

        // check if token exists
        $auth_token = wp_generate_password(32, true, false);  // generates a 64-character random string

        // Get user by email
        $user = get_user_by( 'email', $email );

        if ( $user ) {
            $this->destroy_user_sessions( $user->ID );
            // If the user already exists, just update the token
            update_user_meta( $user->ID, 'oa_auth_token', $auth_token );

            // Update oa_auth_token
            $wpdb->query(
                $wpdb->prepare("UPDATE " . $wpdb->prefix . "oa_woocommerce_persistent_cart SET oa_auth_token = %s WHERE cart_id = %s", $auth_token, $token)
            );


            // update_user_meta( $user->ID, 'oa_last_login', time() );

        } else {
            // If the user does not exist, create a new user
            $random_password = wp_generate_password( 16, true );
            $user_id = $this->createWpUser($email, $random_password);

            // Check if the user was created successfully
            if ( ! is_wp_error( $user_id ) ) {
                $this->destroy_user_sessions( $user_id );
                // Update the token
                update_user_meta( $user_id, 'oa_auth_token', $auth_token );
                // Update oa_auth_token
                $wpdb->query(
                    $wpdb->prepare("UPDATE " . $wpdb->prefix . "oa_woocommerce_persistent_cart SET oa_auth_token = %s WHERE cart_id = %s", $auth_token, $token)
                );

            } else {
                // Handle error
                // For example, return a REST response with an error message
                return new WP_REST_Response( "Unable to create user.", 500 );
            }
        }

        $wpRestResponse = new WP_REST_Response( null, 200 );

        // Calculate X-Server-Authorization and set the header if it exists
        $expectedXServerAuth = $this->calculate_server_authorization($headers, null);

        if($expectedXServerAuth !== null) {
            $this->openappgw_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }

        return $wpRestResponse;
    }

}
