<?php
/**
 * Return Management trait for OpenApp Gateway.
 *
 * @package OpenApp_Payment_Gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Trait OPENAPPGW_Trait_Return_Management
 *
 * Handles order returns processing, admin UI, and OpenApp return sync.
 */
trait OPENAPPGW_Trait_Return_Management {

    public function handle_order_return(WP_REST_Request $request) {
        $context = 'openapp_order_return';

        // Get request data
        $data = $request->get_json_params();
        $headers = $request->get_headers();

        // Log the incoming request
        $this->openappgw_custom_log('Return request received: ' . print_r($data, true), $context);
        $this->openappgw_custom_log('Headers: ' . print_r($this->encodeJsonResponse($headers), true), $context);

        // Validate HMAC
        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if (!$validHmac) {
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        // Validate required fields
        if (empty($data['oaOrderId']) || empty($data['oaOrderReturnId']) || empty($data['shopOrderId'])) {
            return new WP_Error('missing_required_fields', 'Required fields missing', array('status' => 400));
        }

        // Find the WooCommerce order
        $shop_order_id = $data['shopOrderId'];
        $order = wc_get_order($shop_order_id);

        if (!$order) {
            return new WP_Error('order_not_found', 'Order not found', array('status' => 404));
        }

        // Verify the order belongs to this OA order
        $stored_oa_order_id = get_post_meta($shop_order_id, 'oaOrderId', true);
        if ($stored_oa_order_id !== $data['oaOrderId']) {
            return new WP_Error('order_mismatch', 'Order ID mismatch', array('status' => 400));
        }

        // Generate a unique case ID for this return
        $case_id = $this->generate_return_case_id($data);

        // Process the return
        $this->process_order_return($order, $data, $case_id);

        // Prepare response
        $response = array(
            'caseId' => $case_id
        );

        $this->openappgw_custom_log('Return processed successfully. Case ID: ' . $case_id, $context);

        // Create response with proper headers
        $wpRestResponse = new WP_REST_Response($response, 200);

        // Calculate and set X-Server-Authorization header
        $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
        if ($expectedXServerAuth !== null) {
            $this->openappgw_custom_log("X-Server-Authorization: " . $expectedXServerAuth, $context);
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }

        return $wpRestResponse;
    }

    private function generate_return_case_id($return_data) {
        $order_id = $return_data['shopOrderId'];
        $date = date('ymd_His');
        return "OA-RET-{$order_id}-{$date}";
    }

    private function process_order_return($order, $return_data, $case_id) {
        $context = 'openapp_process_return';

        // Store return information as order meta
        $return_meta = array(
            'oa_return_id' => $return_data['oaOrderReturnId'],
            'case_id' => $case_id,
            'return_type' => $this->determine_return_type($return_data['returnedProducts']),
            'status' => 'RECEIVED',
            'return_data' => $return_data,
            'created_at' => current_time('mysql')
        );

        // HPOS-compatible way to store meta
        $order->update_meta_data('_openapp_return_' . $return_data['oaOrderReturnId'], $return_meta);
        $order->save();

        // Create order note with return details
        $note = $this->build_return_note($return_data, $case_id);
        $order->add_order_note($note);

        // Set order status based on return type and settings
        $this->update_order_status_for_return($order, $return_meta['return_type']);

        // Log the return processing
        $this->openappgw_custom_log('Return processed for order ' . $order->get_id(), $context);
    }

    private function determine_return_type($returned_products) {
        $types = array();

        foreach ($returned_products as $product) {
            if (isset($product['returnType'])) {
                $types[] = $product['returnType'];
            }
        }

        // Get unique types and return as comma-separated string
        $unique_types = array_unique($types);
        return implode(', ', $unique_types);
    }

    private function build_return_note($return_data, $case_id) {
        $note = "=== OPENAPP RETURN REQUEST ===\n";
        $note .= "Case ID: " . $case_id . "\n\n";

        $note .= $this->format_data_recursively($return_data);

        return $note;
    }

    private function format_data_recursively($data, $indent = 0) {
        $output = '';
        $spacing = str_repeat('  ', $indent);

        foreach ($data as $key => $value) {
            $label = $this->format_label($key);

            if (is_array($value)) {
                // Handle arrays of objects (like returnedProducts)
                if (isset($value[0]) && is_array($value[0])) {
                    $output .= $spacing . $label . ":\n";
                    foreach ($value as $index => $item) {
                        $output .= $spacing . "  " . ($index + 1) . ".\n";
                        $output .= $this->format_data_recursively($item, $indent + 2);
                        $output .= "\n";
                    }
                } else {
                    // Handle nested objects
                    $output .= $spacing . $label . ":\n";
                    $output .= $this->format_data_recursively($value, $indent + 1);
                }
            } else {
                // Handle single values
                $formatted_value = $this->format_value($key, $value);
                $output .= $spacing . $label . ": " . $formatted_value . "\n";
            }
        }

        return $output;
    }

    private function format_label($key) {
        // Convert camelCase/snake_case to readable labels
        $label = preg_replace('/([a-z])([A-Z])/', '$1 $2', $key);
        $label = str_replace('_', ' ', $label);
        return ucwords($label);
    }

    private function format_value($key, $value) {
        // Handle special formatting for known fields
        if (in_array($key, ['unitPrice', 'linePrice', 'totalPrice'])) {
            return wc_price($value / 100);
        }

        if ($key === 'id' && is_numeric($value)) {
            // Try to get product name for product IDs
            $product = wc_get_product($value);
            if ($product) {
                return $product->get_name() . " (ID: {$value})";
            }
        }

        if (is_bool($value)) {
            return $value ? 'Yes' : 'No';
        }

        if (is_null($value)) {
            return 'N/A';
        }

        return $value;
    }

    private function update_order_status_for_return($order, $return_type) {
        // Get setting for return handling
        $return_status_setting = $this->get_option('return_order_status', 'on-hold');

        // Handle multiple return types
        $types = array_map('trim', explode(',', $return_type));

        if (count($types) === 1) {
            $single_type = $types[0];
            switch ($single_type) {
                case 'CLAIM':
                    $order->update_status($return_status_setting, 'Return claim received - requires investigation');
                    break;
                case 'WITHDRAWAL':
                    $order->update_status($return_status_setting, 'Customer withdrawal request received');
                    break;
                default:
                    $order->update_status($return_status_setting, 'Return request received');
                    break;
            }
        } else {
            // Multiple types
            $message = 'Mixed return request received (' . $return_type . ')';
            $order->update_status($return_status_setting, $message);
        }
    }


    public function add_order_returns_meta_box() {
        $screen = get_current_screen();

        // Add meta box for different possible screen IDs
        $screens = array('shop_order', 'woocommerce_page_wc-orders', 'wc_orders');

        foreach ($screens as $screen_id) {
            add_meta_box(
                'openapp_returns_' . $screen_id,
                'OpenApp Returns',
                array($this, 'display_order_returns_meta_box'),
                $screen_id,
                'normal',
                'high'
            );
        }
    }

    public function display_order_returns_meta_box($post) {
        $order_id = ($post instanceof WC_Order) ? $post->get_id() : $post->ID;
        $returns = $this->get_order_returns($order_id);

        if (empty($returns)) {
            echo '<p>No returns found for this order.</p>';
            return;
        }

        echo '<div class="openapp-returns-container">';
        echo '<style>
.return-status { padding: 4px 8px; border-radius: 3px; font-weight: bold; }
.status-received { background: #ffc107; color: #000; }
.status-accepted { background: #28a745; color: #fff; }
.status-rejected { background: #dc3545; color: #fff; }
.return-actions { margin-top: 15px; }
.return-details { margin: 15px 0; padding: 15px; background: #f9f9f9; border-left: 4px solid #0073aa; }
.return-data-section { margin-top: 15px; padding: 10px; background: #fff; border: 1px solid #ddd; border-radius: 4px; }
.return-data-section h5 { margin-top: 0; color: #333; }
.return-data-content { font-family: monospace; white-space: pre-line; font-size: 12px; line-height: 1.4; }
.openapp-button { margin-right: 10px !important; color: #fff !important; text-decoration: none !important; }
.openapp-button.accept { background: #28a745 !important; border-color: #28a745 !important; }
.openapp-button.reject { background: #dc3545 !important; border-color: #dc3545 !important; }
.openapp-button:hover.accept { background: #218838 !important; border-color: #1e7e34 !important; }
.openapp-button:hover.reject { background: #c82333 !important; border-color: #bd2130 !important; }
</style>';

        foreach ($returns as $return_meta_key => $return) {
            $this->display_single_return($return, $return_meta_key, $order_id);
        }

        echo '</div>';

        // Add JavaScript for AJAX handling
        $this->add_returns_admin_script();
    }

    private function display_single_return($return, $meta_key, $order_id) {
        $return_data = $return['return_data'];

        echo '<div class="return-details">';
        echo '<h4>Return Request: ' . esc_html($return['case_id']) . '</h4>';

        // Basic return info
        echo '<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">';
        echo '<div>';
        echo '<p><strong>Return Type:</strong> ' . esc_html($return['return_type']) . '</p>';
        echo '<p><strong>Status:</strong> <span class="return-status status-' . strtolower($return['status']) . '">' . esc_html($return['status']) . '</span></p>';
        echo '<p><strong>OA Return ID:</strong> ' . esc_html($return['oa_return_id']) . '</p>';
        echo '</div>';
        echo '<div>';
        echo '<p><strong>Created:</strong> ' . esc_html($return['created_at']) . '</p>';
        if (isset($return_data['returnDelivery'])) {
            echo '<p><strong>Return Method:</strong> ' . esc_html($return_data['returnDelivery']['method'] ?? 'N/A') . '</p>';
            if (isset($return_data['returnDelivery']['trackingNumber'])) {
                echo '<p><strong>Tracking:</strong> ' . esc_html($return_data['returnDelivery']['trackingNumber']) . '</p>';
            }
        }
        echo '</div>';
        echo '</div>';

        // Show returned products with more detail
        echo '<div style="margin-bottom: 15px;">';
        echo '<h5>Returned Products:</h5>';
        echo '<table style="width: 100%; border-collapse: collapse;">';
        echo '<thead>';
        echo '<tr style="background: #f0f0f0;">';
        echo '<th style="padding: 8px; text-align: left; border: 1px solid #ddd;">Product</th>';
        echo '<th style="padding: 8px; text-align: center; border: 1px solid #ddd;">Qty</th>';
        echo '<th style="padding: 8px; text-align: center; border: 1px solid #ddd;">Type</th>';
        echo '<th style="padding: 8px; text-align: right; border: 1px solid #ddd;">Unit Price</th>';
        echo '<th style="padding: 8px; text-align: right; border: 1px solid #ddd;">Line Total</th>';
        echo '<th style="padding: 8px; text-align: left; border: 1px solid #ddd;">Reason</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ($return_data['returnedProducts'] as $product) {
            $wc_product = wc_get_product($product['id']);
            $product_name = $wc_product ? $wc_product->get_name() : 'Product ID: ' . $product['id'];

            echo '<tr>';
            echo '<td style="padding: 8px; border: 1px solid #ddd;">' . esc_html($product_name) . '</td>';
            echo '<td style="padding: 8px; text-align: center; border: 1px solid #ddd;">' . esc_html($product['returnedQuantity']) . '</td>';
            echo '<td style="padding: 8px; text-align: center; border: 1px solid #ddd;">' . esc_html($product['returnType'] ?? 'N/A') . '</td>';
            echo '<td style="padding: 8px; text-align: right; border: 1px solid #ddd;">' . wc_price($product['unitPrice'] / 100) . '</td>';
            echo '<td style="padding: 8px; text-align: right; border: 1px solid #ddd;">' . wc_price($product['linePrice'] / 100) . '</td>';
            echo '<td style="padding: 8px; border: 1px solid #ddd;">' . esc_html($product['reasonLabels']['pl'] ?? 'N/A') . '</td>';
            echo '</tr>';

            // Add notes row if exists
            if (!empty($product['notes'])) {
                echo '<tr>';
                echo '<td colspan="6" style="padding: 8px; border: 1px solid #ddd; background: #f9f9f9; font-style: italic;">';  // Changed from 5 to 6
                echo '<strong>Notes:</strong> ' . esc_html($product['notes']);
                echo '</td>';
                echo '</tr>';
            }
        }
        echo '</tbody>';
        echo '</table>';
        echo '</div>';

        // Action buttons for pending returns
        if ($return['status'] === 'RECEIVED') {
            echo '<div class="return-actions">';
            echo '<button type="button" class="button button-primary openapp-button accept accept-return"
          data-return-key="' . esc_attr($meta_key) . '"
          data-order-id="' . esc_attr($order_id) . '"
          data-action="accept">';
            echo '✓ Accept Refund via OpenApp';
            echo '</button>';
            echo '<button type="button" class="button openapp-button reject reject-return"
          data-return-key="' . esc_attr($meta_key) . '"
          data-order-id="' . esc_attr($order_id) . '"
          data-action="reject">';
            echo '✗ Reject Refund Request';
            echo '</button>';
            echo '</div>';
        }

        echo '</div>';
    }


    private function get_order_returns($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return array();
        }

        $returns = array();
        $meta_data = $order->get_meta_data();

        foreach ($meta_data as $meta) {
            $key = $meta->get_data()['key'];
            if (strpos($key, '_openapp_return_') === 0) {
                $returns[$key] = $meta->get_data()['value'];
            }
        }

        return $returns;
    }

    /**
     * Add JavaScript for admin return actions
     */
    private function add_returns_admin_script() {
        ?>
        <script type="text/javascript">
            jQuery(document).ready(function($) {
                $('.accept-return, .reject-return').on('click', function(e) {
                    e.preventDefault();

                    var $button = $(this);
                    var returnKey = $button.data('return-key');
                    var orderId = $button.data('order-id');
                    var action = $button.data('action');
                    var originalText = $button.text();

                    // Define labels based on action
                    var labels = {
                        'accept': {
                            actionText: 'Accept',
                            successMessage: 'Accept successful!',
                            processingLabel: 'Processing...'
                        },
                        'reject': {
                            actionText: 'Reject',
                            successMessage: 'Reject successful!',
                            processingLabel: 'Processing...'
                        }
                    };

                    var currentLabels = labels[action];
                    var notes = prompt('Enter notes for this ' + currentLabels.actionText.toLowerCase() + ' action:');
                    if (notes === null) return; // User cancelled

                    $button.prop('disabled', true).text(currentLabels.processingLabel);

                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'openapp_handle_return_action',
                            return_key: returnKey,
                            order_id: orderId,
                            return_action: action,
                            notes: notes,
                            nonce: '<?php echo wp_create_nonce('openapp_return_action'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                alert(currentLabels.successMessage);
                                location.reload();
                            } else {
                                alert('Error: ' + response.data);
                                $button.prop('disabled', false).text(originalText);
                            }
                        },
                        error: function() {
                            alert('An error occurred. Please try again.');
                            $button.prop('disabled', false).text(originalText);
                        }
                    });
                });
            });
        </script>
        <?php
    }

    public function handle_return_action_ajax() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'openapp_return_action')) {
            wp_send_json_error('Invalid nonce');
        }

        // Check permissions
        if (!current_user_can('edit_shop_orders')) {
            wp_send_json_error('Insufficient permissions');
        }

        $return_key = sanitize_text_field($_POST['return_key']);
        $order_id = intval($_POST['order_id']);
        $action = sanitize_text_field($_POST['return_action']); // 'accept' or 'reject'
        $notes = sanitize_textarea_field($_POST['notes']);

        $order = wc_get_order($order_id);
        if (!$order) {
            wp_send_json_error('Order not found');
        }

        // Get return data - HPOS compatible
        $return_meta = $order->get_meta($return_key, true);
        if (!$return_meta) {
            wp_send_json_error('Return not found');
        }

        $new_status = ($action === 'accept') ? 'ACCEPTED' : 'REJECTED';

        // Update return status locally - HPOS compatible
        $return_meta['status'] = $new_status;
        $return_meta['notes'] = $notes;
        $return_meta['updated_at'] = current_time('mysql');
        $order->update_meta_data($return_key, $return_meta);
        $order->save();

        // Send update to OpenApp
        $oa_response = $this->update_return_status_in_openapp(
            $return_meta['oa_return_id'],
            $new_status,
            $notes,
            (string)$order_id
        );

        if (!$oa_response['success']) {
            // Revert local change if OpenApp update failed - HPOS compatible
            $return_meta['status'] = 'RECEIVED';
            unset($return_meta['notes']);
            unset($return_meta['updated_at']);
            $order->update_meta_data($return_key, $return_meta);
            $order->save();

            wp_send_json_error('Failed to update OpenApp: ' . $oa_response['error']);
        }

        // Process the return in WooCommerce
        if ($new_status === 'ACCEPTED') {
            $this->process_return_acceptance($order, $return_meta, $notes);
        } else {
            $this->process_return_rejection($order, $return_meta, $notes);
        }

        wp_send_json_success('Return ' . strtolower($new_status) . ' successfully');
    }

    private function process_return_acceptance($order, $return_meta, $notes) {
        $return_data = $return_meta['return_data'];

        // Check if auto-refund is enabled
        $auto_refund = $this->get_option('auto_refund_on_accept', 'yes');

        if ($auto_refund === 'yes') {
            $refund_result = $this->create_wc_refund($order, $return_data);

            if ($refund_result['success']) {
                $refund = $refund_result['refund'];
                $order->add_order_note(sprintf(
                    "Return accepted and refund #%d created. Amount: %s. Admin notes: %s",
                    $refund->get_id(),
                    wc_price($refund->get_amount()),
                    $notes
                ));
            } else {
                $order->add_order_note("Return accepted but refund creation failed: " . $refund_result['error'] . ". Manual processing required. Admin notes: " . $notes);
            }
        } else {
            $order->add_order_note("Return accepted. Manual refund processing required. Admin notes: " . $notes);
        }
    }

    private function process_return_rejection($order, $return_meta, $notes) {
        $order->add_order_note("Return rejected by admin. Reason: " . $notes);
    }

    private function create_wc_refund($order, $return_data) {
        try {
            // Calculate refund amount and line items
            $refund_amount = 0;
            $line_items = array();

            foreach ($return_data['returnedProducts'] as $returned_product) {
                $product_id = $returned_product['id'];
                $returned_qty = $returned_product['returnedQuantity'];
                $line_price = $returned_product['linePrice'] / 100; // Convert from cents

                $refund_amount += $line_price;

                // Find the order item for this product
                foreach ($order->get_items() as $item_id => $item) {
                    if ($item->get_product_id() == $product_id || $item->get_variation_id() == $product_id) {
                        $line_items[$item_id] = array(
                            'qty' => $returned_qty,
                            'refund_total' => $line_price,
                            'refund_tax' => 0 // You might want to calculate tax here
                        );
                        break;
                    }
                }
            }

            // Create the refund
            $refund = wc_create_refund(array(
                'order_id' => $order->get_id(),
                'amount' => $refund_amount,
                'reason' => 'OpenApp Return: ' . $return_data['type'] . ' (Return ID: ' . $return_data['oaOrderReturnId'] . ')',
                'line_items' => $line_items
            ));

            if (is_wp_error($refund)) {
                return array(
                    'success' => false,
                    'error' => $refund->get_error_message()
                );
            }

            return array(
                'success' => true,
                'refund' => $refund
            );

        } catch (Exception $e) {
            return array(
                'success' => false,
                'error' => $e->getMessage()
            );
        }
    }



    private function update_return_status_in_openapp($oa_return_id, $status, $notes, $shop_order_id) {
        if (!$oa_return_id) {
            return array(
                'success' => false,
                'error' => 'Return ID is required'
            );
        }

        // Validate status according to API docs
        $valid_statuses = ['RECEIVED', 'ACCEPTED', 'REJECTED'];
        if (!in_array($status, $valid_statuses)) {
            return array(
                'success' => false,
                'error' => 'Status must be one of: ' . implode(', ', $valid_statuses)
            );
        }

        $endpoint = '/v1/returns/' . $oa_return_id;
        $method = 'PATCH';
        $url = $this->open_app_url . $endpoint;
        $context = 'openapp_update_return';

        $start_time = microtime(true);

        // Prepare data for OpenApp (all fields are required according to docs)
        $return_data = array(
            'status' => $status,
            'notes' => $notes,
            'shopOrderId' => $shop_order_id
        );

        $body = wp_json_encode($return_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        $setupData = $this->setupRequest($endpoint, $method, $body, $context);
        $response = $this->executeRequest($url, $setupData, $context);

        $end_time = microtime(true);
        $execution_time = round($end_time - $start_time, 4);
        $this->log_debug("update_return_status_in_openapp: " . $execution_time);

        // Process the actual response
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => $response->get_error_message()
            );
        }

        $http_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        // For PATCH /v1/returns, we expect HTTP 204 (No Content) on success
        if ($http_code === 204) {
            return array(
                'success' => true,
                'status_code' => 204,
                'body' => null,
                'raw_body' => $response_body
            );
        } else {
            // Handle error responses
            $error_message = "HTTP {$http_code}";
            if (!empty($response_body)) {
                $error_data = json_decode($response_body, true);
                if ($error_data && isset($error_data['error'])) {
                    $error_message .= ": " . $error_data['error'];
                } else {
                    $error_message .= ": " . $response_body;
                }
            }

            return array(
                'success' => false,
                'error' => $error_message,
                'status_code' => $http_code,
                'raw_body' => $response_body
            );
        }
    }

}
