<?php

trait OpenappApiClientTrait
{
    /**
     * Send an HMAC-authenticated request to OpenApp API.
     *
     * @param string $endpoint e.g. '/v1/returns/{id}'
     * @param string $method   HTTP method
     * @param array|string $payload Array to JSON-encode or raw JSON string
     * @param string $context  Log context tag
     *
     * @return array {success: bool, status_code: int|null, body: mixed, raw_body: string, error?: string}
     */
    protected function sendOpenAppRequest($endpoint, $method, $payload = array(), $context = 'openapp_request')
    {
        $merchantUrl = Tools::getValue('OA_API_BASE_URL', Configuration::get('OA_API_BASE_URL'));
        $apiKey = Tools::getValue('OA_API_KEY', Configuration::get('OA_API_KEY'));
        $secret = Tools::getValue('OA_API_SECRET', Configuration::get('OA_API_SECRET'));

        if (empty($merchantUrl) || empty($apiKey) || empty($secret)) {
            return array('success' => false, 'status_code' => null, 'error' => 'Missing API configuration');
        }

        $method = strtoupper($method);
        $path = strtoupper($endpoint);
        $timestamp = (int) (microtime(true) * 1000);
        $nonce = $timestamp;

        $body = is_string($payload) ? $payload : $this->encodeJsonResponse($payload);
        $bodyHash = hash('sha256', $body, true);
        $bodyHashBase64 = base64_encode($bodyHash);

        // Use host class' HMAC generator
        $hmacSignature = $this->generateHmacSignature($method, $path, $timestamp, $nonce, $bodyHashBase64, $apiKey, $secret, $context);
        $authHeader = "hmac v1$".$apiKey."$". $method ."$". $path ."$".$timestamp."$".$nonce;

        $headers = array(
            'Content-Type' => 'application/json',
            'authorization' => $authHeader,
            'x-app-signature' => $hmacSignature
        );

        $headersString = '';
        foreach ($headers as $header => $value) {
            $headersString .= $header . ": " . $value . "\r\n";
        }

        $contextOptions = array(
            'http' => array(
                'method'  => $method,
                'header'  => $headersString,
                'content' => $body,
                'ignore_errors' => true
            )
        );

        $contextResource = stream_context_create($contextOptions);

        try {
            $response = Tools::file_get_contents($merchantUrl . $endpoint, false, $contextResource);
        } catch (\Exception $e) {
            return array('success' => false, 'status_code' => null, 'error' => $e->getMessage());
        }

        $status_code = null;
        if (isset($http_response_header) && is_array($http_response_header)) {
            foreach ($http_response_header as $headerLine) {
                if (strpos($headerLine, 'HTTP/') === 0) {
                    $parts = explode(' ', $headerLine);
                    if (count($parts) > 1) {
                        $status_code = intval($parts[1]);
                    }
                    break;
                }
            }
        }

        $success = $status_code !== null && $status_code >= 200 && $status_code < 300;

        $bodyDecoded = json_decode($response, true);
        $bodyToReturn = $bodyDecoded !== null ? $bodyDecoded : $response;

        return array(
            'success' => $success,
            'status_code' => $status_code,
            'body' => $bodyToReturn,
            'raw_body' => $response,
            'error' => $success ? null : $response
        );
    }

    public function generateHmacSignature($method, $path, $timestamp, $nonce, $content = null, $api_key = null, $secret = null, $log_to_context = '')
    {
        if (!$api_key) {
            $api_key = Tools::getValue('OA_API_KEY', Configuration::get('OA_API_KEY'));
        }
        if (!$secret) {
            $secret = Tools::getValue('OA_API_SECRET', Configuration::get('OA_API_SECRET'));
        }

        $stringToSign = "v1$"."$api_key$" . "$method$" . "$path$" . "$timestamp$" . "$nonce";

        if ($content) {
            $stringToSign .= "$".$content;
        }

        $hmacHash = hash_hmac('sha256', $stringToSign, $secret, true);
        $hmacSignature = base64_encode($hmacHash);

        if (method_exists($this, 'ct_custom_log')) {
            $this->ct_custom_log($stringToSign, $log_to_context);
        }

        return $hmacSignature;
    }
}
