<?php

require_once dirname(__FILE__) . '/../classes/OpenappRESTTrait.php';
require_once dirname(__FILE__) . '/../classes/OpenappApiClientTrait.php';

abstract class OpenappAbstractRESTController extends ModuleFrontController
{
    use OpenappRESTTrait;
    use OpenappApiClientTrait;

    private $img1 = 'large';
    private $img2 = 'medium';
    private $img3 = '_default';
    private $secret;
    private $apiKey;

    public function __construct()
    {
        parent::__construct();
        $this->apiKey = Tools::getValue('OA_API_KEY', Configuration::get('OA_API_KEY'));
        $this->secret = Tools::getValue('OA_API_SECRET', Configuration::get('OA_API_SECRET'));

    }

    public function init()
    {

        header('Content-Type: ' . "application/json");

        if (Tools::getValue('iso_currency')){
            $_GET['id_currency'] = (string)Currency::getIdByIsoCode(Tools::getValue('iso_currency'));
            $_GET['SubmitCurrency'] = "1";
        }

        parent::init();

        $response = [
            'success' => true,
            'code' => 210,
            'psdata' => null,
            'message' => 'empty'
        ];

        switch ($_SERVER['REQUEST_METHOD']) {
            case 'GET':
                $response = $this->processGetRequest();
                break;
            case 'POST':
                $response = $this->processPostRequest();
                break;
            case 'PATCH':
            case 'PUT':
                $response = $this->processPutRequest();
                break;
            case 'DELETE':
                $response = $this->processDeleteRequest();
                break;
            default:
                // throw some error or whatever
        }

        $this->ajaxRender(json_encode($response));
        die;
    }

    public function formatPrice($price)
    {
        return Tools::displayPrice(
            $price,
            $this->context->currency,
            false,
            $this->context
        );
    }

    public function getImageType($type = 'large')
    {
        if ($type == 'large') {
            return $this->img1 . $this->img3;
        } elseif ($type == 'medium') {
            return $this->img2 . $this->img3;
        } else {
            return $this->img1 . $this->img3;
        }
    }

    protected function checkCartProductsMinimalQuantities()
    {
        $productList = $this->context->cart->getProducts();

        foreach ($productList as $product) {
            if ($product['minimal_quantity'] > $product['cart_quantity']) {
                // display minimal quantity warning error message
                $this->errors[] = $this->trans(
                    'The minimum purchase order quantity for the product %product% is %quantity%.',
                    [
                        '%product%' => $product['name'],
                        '%quantity%' => $product['minimal_quantity'],
                    ],
                    'Shop.Notifications.Error'
                );
            }
        }
    }


    /**
     * ===========================================================================
     * 9. HMAC Authentication
     * ===========================================================================
     */
    public function hmacSignatureIsValid($authorizationHeader, $serverSignature, $responseBodyHashBase64 = null)
    {
        $context = 'openapp_hmac_response';
        list($version, $api_key, $method, $path, $timestamp, $nonce) = explode('$', $authorizationHeader);

        // Check if the timestamp is within the allowed 60 seconds window
        $currentTimestamp = round(microtime(true) * 1000); // Current time in milliseconds
        $diff = abs($currentTimestamp - $timestamp);

        if ($diff > 60000) {
             $this->ct_custom_log( "hmacSignatureIsValid failed on diff: ". var_export($diff, true) , $context);
             return false; // Reject if the timestamp difference is greater than 60 seconds
        }

        $hmacSignature = $this->generateHmacSignature($method, $path, $timestamp, $nonce, $responseBodyHashBase64, $this->apiKey, $this->secret);

        $this->ct_custom_log("--------------------- start ------------------");
        $this->ct_custom_log( "hmacSignatureIsValid authorizationHeader: ". var_export($authorizationHeader, true) , $context);
        $this->ct_custom_log("hmacSignatureIsValid  hmacSignature: ". var_export($hmacSignature, true), $context);
        $this->ct_custom_log("--------------------- end ------------------");

        if(hash_equals($serverSignature, $hmacSignature)){
            return true;
        } else {
            return false;
        }
    }


    /**
     *
     * controller basket.php
     * L:68 - $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
     *  */
    public function calculate_server_authorization($headers, $responseBody = null) {
        $authorization = $this->getNormalizedHeaderValue($headers, 'Authorization');

        if(!is_null($authorization)) {
            $components = explode('$', $authorization);

            if (count($components) < 6) {
                // The authorization string doesn't contain as many components as we're expecting.
                return null;
            }

            $timestamp = $components[4];
            $nonce = $components[5];

            $calculatedSignature = $this->generateHmacResponseSignature($timestamp, $nonce, $responseBody, $authorization);

            $context = 'openapp_hmac_response';
            $this->ct_custom_log("calculatedSignature: ". $calculatedSignature, $context);

            return "hmac v1". "$"."$timestamp"."$".$nonce."$"."$calculatedSignature";
        }

        return null;
    }


    public function generateHmacResponseSignature($timestamp, $nonce, $responseBody = null, $authorization = null, $printDebug = false)
    {
        $stringToSign =  "v1$"."$timestamp"."$"."$nonce";

        if (!is_null($responseBody)) {
            $responseBody = json_encode($responseBody);

            $responseBodyHash = hash('sha256', $responseBody, true);

            $responseBodyHashBase64 = base64_encode($responseBodyHash);
            $stringToSign .= "$".$responseBodyHashBase64;
        }


        $context = 'openapp_hmac_response';
        $this->ct_custom_log("GENERATE_HMAC_RESPONSE_SIGNATURE", $context);
        $this->ct_custom_log("Request Authorization Header: ". $authorization, $context);
        $this->ct_custom_log("Timestamp: ".$timestamp, $context);
        $this->ct_custom_log("Nonce: ".$nonce, $context);
        $this->ct_custom_log("responseBody2:", $context);
        $this->ct_custom_log(var_export($responseBody, true), $context);
        if(isset($responseBodyHash)){
            $this->ct_custom_log("responseHash: ".$responseBodyHash, $context);
        }
        if(isset($responseBodyHashBase64)){
            $this->ct_custom_log("responseBodyHashBase64: ".$responseBodyHashBase64, $context);
        }

        $this->ct_custom_log("stringToSign: ".$stringToSign, $context);
        $this->ct_custom_log("-----------------------", $context);

        $hmacHash = hash_hmac('sha256', $stringToSign, $this->secret, true);
        $hmacHashBase64 = base64_encode($hmacHash);

        if($printDebug){
            // Output
            var_dump("GENERATING X-Server-Authorization Header");
            var_dump("Secret: ". $this->mask_secret($this->secret));
            var_dump("bodyDigest: " . $responseBodyHashBase64);
            var_dump("Nonce: " . $nonce);
            var_dump("Signature: " . $hmacHashBase64 );
            var_dump("Timestamp: " . $timestamp);
            var_dump("StringToSign: " . $stringToSign);
        }

        return $hmacHashBase64;
    }

    /**
     * PATCH /v1/returns/{id} helper (WP parity)
     */
    protected function update_return_status_in_openapp($oa_return_id, $status, $notes, $shop_order_id)
    {
        if (!$oa_return_id) {
            return array('success' => false, 'error' => 'Return ID is required');
        }

        $valid_statuses = array('RECEIVED', 'ACCEPTED', 'REJECTED');
        if (!in_array($status, $valid_statuses)) {
            return array('success' => false, 'error' => 'Status must be one of: ' . implode(', ', $valid_statuses));
        }

        $payload = array(
            'status' => $status,
            'notes' => $notes,
            'shopOrderId' => (string) $shop_order_id,
        );

        $apiResponse = $this->sendOpenAppRequest('/v1/returns/' . $oa_return_id, 'PATCH', $payload, 'openapp_update_return');

        if ($apiResponse['status_code'] === 204) {
            return array('success' => true, 'status_code' => 204, 'body' => null);
        }

        return array(
            'success' => $apiResponse['success'],
            'status_code' => $apiResponse['status_code'],
            'error' => isset($apiResponse['error']) ? $apiResponse['error'] : 'Unknown error',
            'raw_body' => isset($apiResponse['raw_body']) ? $apiResponse['raw_body'] : null
        );
    }

    private function mask_secret($secret){
        $len = strlen($secret);

        if ($len < 8) {
            // Handle short strings here, e.g., return a fully masked string or the original string
            return str_repeat('*', $len);
        }


        $start = substr($secret, 0, 4); // Get the first four characters
        $end = substr($secret, -4); // Get the last four characters
        $masked = str_repeat('*', strlen($secret) - 8); // Generate a string of '*' with the same length as the remaining characters

        return $start . $masked . $end; // Return the masked secret
    }

    private function getNormalizedHeaderValue(array $headers, $headerName)
    {
        $headerName = str_replace('_', '-', strtolower($headerName));
        $headers = array_combine(
            array_map(function ($key) {
                return str_replace('_', '-', strtolower($key));
            }, array_keys($headers)),
            $headers
        );

        if (isset($headers[$headerName])) {
            // Check if the header's value is an array and return the first element
            if (is_array($headers[$headerName])) {
                return $headers[$headerName][0];
            }
            // The header's value is a string, so return it
            return $headers[$headerName];
        }

        return null;
    }

    public function isRequestValid($headers , $body = null)
    {
        $authorization = $this->getNormalizedHeaderValue($headers, 'Authorization');
        $xAppSignature = $this->getNormalizedHeaderValue($headers, 'X-App-Signature');

        if ($authorization === null || $xAppSignature === null) {
            return false;
        }

        // Retrieve the configuration setting
        // SECURITY: Never enable OA_DISABLE_VALIDATION_MODE in production!
        $isValidationDisabled = Configuration::get('OA_DISABLE_VALIDATION_MODE', false);
        // If validation is disabled, return true immediately
        if ($isValidationDisabled) {
            return true;
        }

        $validHmac = $this->hmacSignatureIsValid( $authorization, $xAppSignature, $body);

        $context = 'openapp_hmac_response';
        $this->ct_custom_log("--------------------- isRequestValid start ------------------");
        $this->ct_custom_log(var_export($headers, true));
        $this->ct_custom_log( "isRequestValid Auth: ". var_export($authorization, true) , $context);
        $this->ct_custom_log("isRequestValid X-App-Signature: ". var_export($xAppSignature, true), $context);
        $this->ct_custom_log("isRequestValid Body: ". var_export($body, true), $context);
        $this->ct_custom_log("isRequestValid: ". var_export($validHmac, true), $context);
        $this->ct_custom_log("--------------------- isRequestValid end ------------------");

        return $validHmac;
    }


    public function ct_custom_log($message, $context = '')
    {
        if (Configuration::get('OA_ENABLE_LOG', false)) {
            // Configure the logger
            $logger = new \Monolog\Logger('oa1');
            $logPath = _PS_ROOT_DIR_.'/var/logs/oadebug.log'; // Define the log path
            $logStreamHandler = new \Monolog\Handler\StreamHandler($logPath, \Monolog\Logger::DEBUG);
            $logger->pushHandler($logStreamHandler);

            // Log the message (you can change the level from debug to info, notice, warning, etc.)
            $logger->debug($message);
        }

        return false;
    }


    public function encodeJsonResponse($response)
    {

        return json_encode($response, JSON_UNESCAPED_SLASHES);
    }

    /**
     * Send standardized error response (WP-compatible format)
     *
     * Returns JSON in format:
     * {
     *   "code": "error_code",
     *   "message": "Human-readable message",
     *   "data": {"status": HTTP_STATUS_CODE}
     * }
     *
     * @param string $code Error code (e.g., 'basket_not_found', 'invalid_auth')
     * @param string $message Human-readable error message
     * @param int $httpStatus HTTP status code (400, 403, 404, 500)
     */
    protected function sendError($code, $message, $httpStatus = 400)
    {
        http_response_code($httpStatus);
        $this->ajaxRender($this->encodeJsonResponse([
            'code' => $code,
            'message' => $message,
            'data' => ['status' => $httpStatus]
        ]));
        die;
    }


    public function get_values_and_types($data) {
        if(is_array($data) || is_object($data)) {
            foreach($data as $key => $value) {
                if(is_array($value) || is_object($value)) {
                    $data[$key] = $this->get_values_and_types($value); // Recursive call for nested array or objects
                } else {
                    $data[$key] = array('value' => $value, 'type' => gettype($value));
                }
            }
        } else {
            $data = array('value' => $data, 'type' => gettype($data));
        }
        return $data;
    }


    public function groszeFormat($floatValue)
    {
        $intValue = (int)round($floatValue * 100); // Multiply by 100 to move the decimal point
        return $intValue;
    }

    /**
     * Format measurement values (weight/dimensions) similar to WP output:
     * - Return empty string for 0/empty
     * - Trim trailing zeros while keeping a decimal point if needed.
     */
    protected function formatMeasurement($value)
    {
        if ($value === '' || $value === null) {
            return '';
        }

        $floatVal = (float)$value;
        if ($floatVal == 0.0) {
            return '';
        }

        $formatted = sprintf('%.6f', $floatVal);
        $formatted = rtrim(rtrim($formatted, '0'), '.');

        return $formatted;
    }


    public function getOAShipping($carriers, $cartTotalPrice, $id_zone)
    {

        $shipping = array();

        if ($carriers && count($carriers) > 0) {
            foreach ($carriers as $carrier) {
                $mappedKey = Configuration::get('CARRIER_MAP_OA_' . $carrier['id_carrier']);

                if ($mappedKey !== false && !empty($mappedKey)) {
                    $shippingCost = $carrier['price'];
                    $shippingCostCents = $this->groszeFormat($shippingCost);

                    if(is_int($shippingCostCents)){
                        $shipping[] = array(
                            'key' => $mappedKey,
                            'cost' => $shippingCostCents
                        );
                    }
                }
            }
        }

        return $shipping;
    }



    public function buildBasketArray($basket) {
        $basketArray = array(
            "expiresAt" => gmdate('Y-m-d\TH:i:s\Z', strtotime('+1 days')),
            "price" => array(
                "currency" => $this->getCurrency(),
                "discounts" => array(), // Add any applicable discounts
                "basketValue" => 0 // Initialize basket value
            ),
            "deliveryOptions" => array(),
            "products" => array(),
            "loggedUser" => (string) ($basket['customer_id'] .'_'. $basket['guest_id'])
        );


        $roundType = Configuration::get('PS_ROUND_TYPE');

        if (method_exists(Context::getContext(), 'getComputingPrecision')) {
            $computingPrecision = Context::getContext()->getComputingPrecision();
        } else {
            $currency = Context::getContext()->currency;
            $computingPrecision = isset($currency->decimals) ? _PS_PRICE_DISPLAY_PRECISION_ : 2;
        }
        
        // 1 = Round on each item
        // 2 = Round on each line
        // 3 = Round on the total

        $totalPrice = 0;
        foreach ($basket['cart_contents']['products'] as $productItem) { // Access as array
            $productId = $productItem['id'];
            $productAttributeId = $productItem['id_product_attribute'];

            $product = new Product($productId, true, Context::getContext()->language->id);
            $productAttributeId = (int) $productAttributeId; // Cast to int, 0 if not set

            if (Validate::isLoadedObject($product)) {
                $productName = $product->name;
                // $productPrice = $product->getPrice();
                $productPrice = Product::getPriceStatic($productId, true, $productAttributeId);
                $ean13 = $product->ean13;

                // If it's a combination, optionally append attributes to the name
                if ($productAttributeId > 0) {
                    $combination = new Combination($productAttributeId);
                    if (Validate::isLoadedObject($combination)) {
                        if($combination->ean13){
                            $ean13 = $combination->ean13;
                        }
                        $attributes = $combination->getAttributesName(Context::getContext()->language->id);
                        $attributesString = implode(', ', array_column($attributes, 'name'));
                        $productName .= " - " . $attributesString; // Append attributes to the name
                    }
                }

                /**
                 * Presta rounding
                 */

                $linePrice = $productPrice * $productItem['quantity'];

                switch ($roundType) {
                    case Order::ROUND_TOTAL:
                        // No rounding here; rounding will be applied to total
                        break;
                    case Order::ROUND_LINE:
                        $linePrice = Tools::ps_round($linePrice, $computingPrecision);
                        break;
                    case Order::ROUND_ITEM:
                    default:
                        $productPrice = Tools::ps_round($productPrice, $computingPrecision);
                        $linePrice = $productPrice * $productItem['quantity'];
                        break;
                }

                $totalPrice += $linePrice;

                $imageLink = '';

                // Check if it's a combination and try to get the combination specific image
                if ($productAttributeId > 0) {
                    $combinationImages = $product->getCombinationImages(Context::getContext()->language->id);
                    if (isset($combinationImages[$productAttributeId]) && count($combinationImages[$productAttributeId])) {
                        $imageId = $combinationImages[$productAttributeId][0]['id_image']; // Get the first image ID of the combination
                        $imageLink = $this->context->link->getImageLink($product->link_rewrite, $imageId, 'home_default');
                    }
                }

                // If no specific image for the combination or it's not a combination
                if ($imageLink == '') {
                    $cover = Product::getCover($productId);
                    if ($cover) {
                        $imageId = $cover['id_image'];
                        $imageLink = $this->context->link->getImageLink($product->link_rewrite, $imageId, 'home_default');
                    }
                }

                $productIdWithIdAttribute = (string) $productId . "_" . $productAttributeId;

                $basketArray['products'][] = array(
                    "ean" => $ean13,
                    "id" => $productIdWithIdAttribute,
                    "name" => $productName,
                    "images" => array($imageLink),
                    "quantity" => intval($productItem['quantity']),
                    "unitPrice" => $this->groszeFormat($productPrice),
                    "linePrice" => $this->groszeFormat($linePrice),
                    "originalUnitPrice" => $this->groszeFormat($productPrice),
                    "originalLinePrice" => $this->groszeFormat($linePrice)
                );
            }
        }

        // Apply rounding on the total if required
        if ($roundType == Order::ROUND_TOTAL) {
            $totalPrice = Tools::ps_round($totalPrice, $computingPrecision);
        }

        $basketArray['price']['basketValue'] = $this->groszeFormat($totalPrice);

        return $basketArray;
    }

    public function getCurrency()
    {
        // Get the default currency ID from PrestaShop configuration
        $defaultCurrencyId = (int)Configuration::get('PS_CURRENCY_DEFAULT');

        // Load the currency object using the default currency ID
        $currency = new Currency($defaultCurrencyId);

        // Return the ISO code of the currency
        return $currency->iso_code;
    }

    /**
     * Format product/combination identifier consistently (productId_combinationId).
     */
    protected function formatProductId($productId, $productAttributeId = 0)
    {
        return (int)$productId . '_' . (int)$productAttributeId;
    }

    /**
     * Parse product/combination identifier into ints (productId, combinationId).
     */
    protected function parseProductId($idString)
    {
        $parts = explode('_', (string)$idString);
        $productId = isset($parts[0]) ? (int)$parts[0] : 0;
        $combinationId = isset($parts[1]) ? (int)$parts[1] : 0;
        return [$productId, $combinationId];
    }

    public function getStoredCartData($session_cart_id) {
        $table_name = _DB_PREFIX_.'ps_oa_persistent_cart';
        $session_cart_id_safe = pSQL($session_cart_id);
        $hashed_session_id = hash('md5', $session_cart_id);

        $sql = "SELECT `cart_contents` FROM `".$table_name."` WHERE `cart_id` = '$session_cart_id_safe' AND `cart_session_id` = '$hashed_session_id'";
        $result = Db::getInstance()->getRow($sql);

        if ($result && isset($result['cart_contents'])) {
            return json_decode($result['cart_contents'], true);
        }

        return false;
    }

    /**
     * ===========================================================================
     * Private Order Messages (Customer Service)
     * ===========================================================================
     */

    /**
     * Add a private admin message to an order's customer service thread.
     * These messages appear in Customer Service section, visible only to admins.
     *
     * @param Order $order The order to add the message to
     * @param string $message The message content
     * @return bool True on success, false on failure
     */
    protected function addPrivateOrderMessage($order, $message)
    {
        $idThread = $this->getOrCreateCustomerThread($order);
        if (!$idThread) {
            $this->ct_custom_log('Failed to get/create customer thread for order ' . $order->id);
            return false;
        }

        $cm = new CustomerMessage();
        $cm->id_customer_thread = (int) $idThread;
        $cm->id_employee = $this->getDefaultEmployeeId();
        $cm->message = $message;
        $cm->private = 1;
        $cm->read = 0;

        if (!$cm->add()) {
            $this->ct_custom_log('Failed to add message: ' . Db::getInstance()->getMsgError());
            return false;
        }
        return true;
    }

    /**
     * Get or create a CustomerThread for an order.
     *
     * @param Order $order The order
     * @return int|null Thread ID or null on failure
     */
    protected function getOrCreateCustomerThread($order)
    {
        $sql = new DbQuery();
        $sql->select('id_customer_thread');
        $sql->from('customer_thread');
        $sql->where('id_order = ' . (int) $order->id);
        $sql->where('id_customer = ' . (int) $order->id_customer);

        $existing = Db::getInstance()->getValue($sql);
        if ($existing) {
            return (int) $existing;
        }

        $customer = new Customer((int) $order->id_customer);

        $ct = new CustomerThread();
        $ct->id_shop = (int) $order->id_shop;
        $ct->id_lang = (int) $order->id_lang;
        $ct->id_contact = 0;
        $ct->id_customer = (int) $order->id_customer;
        $ct->id_order = (int) $order->id;
        $ct->id_product = 0;
        $ct->status = 'open';
        $ct->email = $customer->email ?: '';
        $ct->token = Tools::passwdGen(12);

        if (!$ct->add()) {
            $this->ct_custom_log('Failed to create customer thread: ' . Db::getInstance()->getMsgError());
            return null;
        }

        return (int) $ct->id;
    }

    /**
     * Get default employee ID for messages (first active employee).
     *
     * @return int Employee ID
     */
    protected function getDefaultEmployeeId()
    {
        if ($this->context->employee && $this->context->employee->id) {
            return (int) $this->context->employee->id;
        }
        $sql = new DbQuery();
        $sql->select('id_employee');
        $sql->from('employee');
        $sql->where('active = 1');
        $sql->orderBy('id_employee ASC');
        $employeeId = (int) Db::getInstance()->getValue($sql);
        return $employeeId > 0 ? $employeeId : 1;
    }

}
