<?php
require_once dirname(__FILE__) . '/../OpenappAbstractRESTController.php';

use PrestaShop\PrestaShop\Adapter\Presenter\Cart\CartPresenter;
use Symfony\Component\HttpFoundation\Request;

/**
 * This REST endpoint displays order information
 */
class ps_openappbasketModuleFrontController extends OpenappAbstractRESTController
{

    /**
     * @var string Country ISO Code
     */
    private $countryIsoCode = 'PL';

    protected function processGetRequest()
    {
        header('Content-Type: application/json; charset=utf-8');


        if( isset($_GET['basketId']) && !is_null( $_GET['basketId']) ) {

            $basketId = $_GET['basketId'];
            $lang_id         = (int) Configuration::get('PS_LANG_DEFAULT');
            $request = Request::createFromGlobals();
            $headers = $request->headers->all();

            $context = 'openapp_retrieve_basket';
            $this->ct_custom_log("Basket ID: ".print_r($basketId, true), $context);
            $this->ct_custom_log("Headers 2: ".print_r($this->encodeJsonResponse($headers), true), $context); // Logs the headers

            // Verify HMAC
            $validHmac = $this->isRequestValid($headers);

            if(!$validHmac)
            {
                $this->sendError('invalid_auth', $this->trans('Unauthorized request', [], 'Modules.Openapp.Cart'), 403);
            }

            $basket = $this->getStoredCartData($basketId);

            // returns basket_not_found
            if( !$basket )
            {
                $this->sendError('basket_not_found', $this->trans('Basket not found', [], 'Modules.Openapp.Cart'), 404);
            }


            $basketData = $this->buildBasketArray($basket);

            // Apply cart discounts from PrestaShop Cart (same logic as qrcode.php)
            $cart = new Cart($basketId);
            if (Validate::isLoadedObject($cart)) {
                $cartRules = $cart->getCartRules();
                $totalDiscountValue = 0;

                foreach ($cartRules as $cartRule) {
                    $discountValue = isset($cartRule['value_real']) ? (float)$cartRule['value_real'] : 0;
                    if ($discountValue > 0) {
                        $basketData['price']['discounts'][] = [
                            'code' => isset($cartRule['code']) && !empty($cartRule['code']) ? $cartRule['code'] : $cartRule['name'],
                            'value' => $this->groszeFormat($discountValue)
                        ];
                        $totalDiscountValue += $discountValue;
                    }
                }

                if ($totalDiscountValue > 0) {
                    $basketData['price']['basketValue'] = max(0, $basketData['price']['basketValue'] - $this->groszeFormat($totalDiscountValue));
                }
            }

            /**
             * Get shipping methods
             */

            $countryId = Country::getByIso($this->countryIsoCode);
            if (!$countryId) {
                $this->sendError('country_not_found', $this->trans('Country PL not exists', [], 'Modules.Openapp.Cart'), 404);
            }

            $country = new Country($countryId);
            $zoneId = $country->id_zone;

            // Load the cart object
            $cart = new Cart($basketId);
            $customer = new Customer($cart->id_customer); // 0 - guest (?)
            $customerGroups = $customer->getGroups();
            $carrier_error = null;
            $carriers = Carrier::getCarriersForOrder($zoneId, $customerGroups, $cart, $carrier_error);

            // Get total price of products in the cart
            // $cartTotalPrice = $cart->getOrderTotal(true, Cart::ONLY_PRODUCTS);
            $cartTotalPriceCents = $basketData['price']['basketValue'];
            $cartTotalPrice = $cartTotalPriceCents / 100;

            // Calculate the shipping cost based on price
            $basketData["deliveryOptions"] = $this->getOAShipping($carriers, $cartTotalPrice, $zoneId);

            if ($carrier_error) {
                // array of exceptions
                // @TODO - maybe handle those errors (?)
            }

            //Prepare the response
            $response = array(
                'id' => $basketId,
            );
            $response = array_merge($response, $basketData);

            $this->ct_custom_log("responseBody: ". var_export($response, true), $context);

            // Calculate X-Server-Authorization and set the header if it exists
            $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);

            if($expectedXServerAuth !== null) {
                $this->ct_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
                header('X-Server-Authorization: '.$expectedXServerAuth);
            }

            $this->ajaxRender(json_encode($response));
            die;

        } // end $_GET

        $this->sendError('basketid_missing', $this->trans('basketId param missing', [], 'Modules.Openapp.Cart'), 400);

    }

    /**
     * Process POST request - create or update basket with products
     */
    protected function processPostRequest()
    {
        header('Content-Type: application/json; charset=utf-8');

        $context = 'openapp_create_basket';
        $bodyRaw = Tools::file_get_contents('php://input');
        $_POST = json_decode($bodyRaw, true);

        $request = Request::createFromGlobals();
        $headers = $request->headers->all();

        $this->ct_custom_log("POST /basket called", $context);
        $this->ct_custom_log("Body: " . $bodyRaw, $context);

        // Verify HMAC with body hash
        $bodyHash = hash('sha256', $bodyRaw, true);
        $responseBodyHashBase64 = base64_encode($bodyHash);
        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if (!$validHmac) {
            $this->sendError('invalid_auth', $this->trans('Unauthorized request', [], 'Modules.Openapp.Cart'), 403);
        }

        // Validate request structure
        if (!is_array($_POST) || !isset($_POST['basket'])) {
            $this->sendError('invalid_data', $this->trans('Invalid JSON structure - basket required', [], 'Modules.Openapp.Cart'), 400);
        }

        $basketData = $_POST['basket'];
        $products = isset($basketData['products']) ? $basketData['products'] : [];
        $providedBasketId = isset($basketData['id']) ? $basketData['id'] : null;
        $loggedUser = isset($basketData['loggedUser']) ? $basketData['loggedUser'] : null;

        $langId = (int) Configuration::get('PS_LANG_DEFAULT');

        // === Virtual basket calculation (no DB persistence) ===
        // If caller didn't provide an ID, generate one (no persistence implied)
        $basketId = $providedBasketId ?: uniqid();
        $outputProducts = [];
        $basketValueFloat = 0;

        foreach ($products as $product) {
            if (!isset($product['id']) || !isset($product['quantity'])) {
                continue;
            }

            // Parse product ID (format: "productId_attributeId")
            list($productId, $productAttributeId) = $this->parseProductId($product['id']);
            $quantity = (int) $product['quantity'];

            if ($productId <= 0 || $quantity <= 0) {
                continue;
            }

            // Validate product exists and is active
            $productObj = new Product($productId, true, $langId);
            if (!Validate::isLoadedObject($productObj) || !$productObj->active) {
                continue;
            }

            // Stock/availability checks to mirror WP behavior
            $error = null;
            $availableQuantity = StockAvailable::getQuantityAvailableByProduct($productId, $productAttributeId);
            if ($availableQuantity <= 0) {
                $error = 'OUT_OF_STOCK';
                $quantity = 0;
            } elseif ($quantity > $availableQuantity) {
                $quantity = $availableQuantity;
                $error = 'QUANTITY_TOO_BIG';
            }

            $unitPrice = Product::getPriceStatic($productId, true, $productAttributeId);
            $linePrice = $unitPrice * $quantity;
            $basketValueFloat += $linePrice;

            // Get image
            $imageLink = '';
            if ($productAttributeId > 0) {
                $combinationImages = $productObj->getCombinationImages($langId);
                if (isset($combinationImages[$productAttributeId]) && count($combinationImages[$productAttributeId])) {
                    $imageId = $combinationImages[$productAttributeId][0]['id_image'];
                    $imageLink = Context::getContext()->link->getImageLink($productObj->link_rewrite, $imageId, 'home_default');
                }
            }
            if ($imageLink == '') {
                $cover = Product::getCover($productId);
                if ($cover) {
                    $imageLink = Context::getContext()->link->getImageLink($productObj->link_rewrite, $cover['id_image'], 'home_default');
                }
            }

            // EAN (prefer combination)
            $ean = $productObj->ean13;
            if ($productAttributeId > 0) {
                $combination = new Combination($productAttributeId);
                if (Validate::isLoadedObject($combination) && $combination->ean13) {
                    $ean = $combination->ean13;
                }
            }

            // Name with attributes
            $productName = $productObj->name;
            if ($productAttributeId > 0) {
                $combination = new Combination($productAttributeId);
                if (Validate::isLoadedObject($combination)) {
                    $attributes = $combination->getAttributesName($langId);
                    $attributesString = implode(', ', array_column($attributes, 'name'));
                    $productName .= " - " . $attributesString;
                }
            }

            $outputProducts[] = [
                'ean' => $ean,
                'id' => $this->formatProductId($productId, $productAttributeId),
                'name' => $productName,
                'images' => [$imageLink],
                'quantity' => $quantity,
                'unitPrice' => $this->groszeFormat($unitPrice),
                'linePrice' => $this->groszeFormat($linePrice),
                'originalUnitPrice' => $this->groszeFormat($unitPrice),
                'originalLinePrice' => $this->groszeFormat($linePrice)
            ];
            if ($error !== null) {
                $outputProducts[count($outputProducts) - 1]['error'] = $error;
            }
        }

        $basketValueCents = $this->groszeFormat($basketValueFloat);

        // Shipping options (manual calculation, no persisted cart)
        $deliveryOptions = [];
        $countryId = Country::getByIso($this->countryIsoCode);
        if ($countryId) {
            $country = new Country($countryId);
            $zoneId = $country->id_zone;
            $cartTotalPrice = $basketValueCents / 100;
            $carriers = Carrier::getCarriers($langId, true, false, false, null, Carrier::ALL_CARRIERS);

            foreach ($carriers as $carrier) {
                $mappedKey = Configuration::get('CARRIER_MAP_OA_' . $carrier['id_carrier']);
                if ($mappedKey !== false && !empty($mappedKey)) {
                    $carrierObj = new Carrier($carrier['id_carrier']);
                    $shippingCost = 0;
                    if ($carrier['is_free'] == 1) {
                        $shippingCost = 0;
                    } else {
                        $shippingCost = $carrierObj->getDeliveryPriceByPrice($cartTotalPrice, $zoneId);
                        if ($shippingCost === false) {
                            $shippingCost = 0;
                        }
                    }

                    $deliveryOptions[] = [
                        'key' => $mappedKey,
                        'cost' => $this->groszeFormat($shippingCost)
                    ];
                }
            }
        }

        $response = [
            'id' => $basketId,
            'expiresAt' => gmdate('Y-m-d\TH:i:s\Z', strtotime('+1 day')),
            'price' => [
                'currency' => $this->getCurrency(),
                'discounts' => [],
                'basketValue' => $basketValueCents
            ],
            'deliveryOptions' => $deliveryOptions,
            'products' => $outputProducts,
            'loggedUser' => $loggedUser
        ];

        $this->ct_custom_log("Response: " . json_encode($response), $context);

        // Calculate X-Server-Authorization
        $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
        if ($expectedXServerAuth !== null) {
            header('X-Server-Authorization: ' . $expectedXServerAuth);
        }

        http_response_code(200);
        $this->ajaxRender(json_encode($response));
        die;
    }

    /**
     * Get existing cart or create a new one
     */
    private function getOrCreateCart($existingBasketId, $loggedUser)
    {
        $cart = null;

        // Try to load existing cart
        if ($existingBasketId) {
            $cart = new Cart($existingBasketId);
            if (Validate::isLoadedObject($cart)) {
                return $cart;
            }
        }

        // Create new cart
        $cart = new Cart();
        $cart->id_lang = (int) Configuration::get('PS_LANG_DEFAULT');
        $cart->id_currency = (int) Configuration::get('PS_CURRENCY_DEFAULT');
        $cart->id_shop = (int) Context::getContext()->shop->id;
        $cart->id_shop_group = (int) Context::getContext()->shop->id_shop_group;

        // Handle logged user
        if ($loggedUser) {
            $parts = explode('_', $loggedUser);
            if (count($parts) >= 2) {
                $customerId = (int) $parts[0];
                $guestId = (int) $parts[1];

                if ($customerId > 0) {
                    $customer = new Customer($customerId);
                    if (Validate::isLoadedObject($customer)) {
                        $cart->id_customer = $customerId;
                        $cart->secure_key = $customer->secure_key;
                    }
                }
                if ($guestId > 0) {
                    $cart->id_guest = $guestId;
                }
            }
        }

        // If no customer, create as guest cart
        if (!$cart->id_customer) {
            $cart->id_customer = 0;
            $cart->id_guest = 0;
        }

        $cart->add();
        return $cart;
    }

    /**
     * Add a product to cart
     *
     * @param Cart $cart
     * @param array $product Product data with 'id' and 'quantity'
     * @param int $langId
     * @return true|string True on success, error message on failure
     */
    private function addProductToCart($cart, $product, $langId)
    {
        if (!isset($product['id']) || !isset($product['quantity'])) {
            return "Product missing id or quantity";
        }

        // Parse product ID (format: "productId_attributeId")
        $idParts = explode('_', $product['id']);
        $productId = (int) $idParts[0];
        $productAttributeId = isset($idParts[1]) ? (int) $idParts[1] : 0;
        $quantity = (int) $product['quantity'];

        if ($productId <= 0 || $quantity <= 0) {
            return "Invalid product ID or quantity";
        }

        // Validate product
        $productObj = new Product($productId, false, $langId);
        if (!Validate::isLoadedObject($productObj) || !$productObj->active || !$productObj->available_for_order) {
            return "Product ID {$productId} is not valid or available";
        }

        // Check stock
        $availableQuantity = StockAvailable::getQuantityAvailableByProduct($productId, $productAttributeId);
        if ($availableQuantity < $quantity && !$productObj->out_of_stock) {
            // Adjust quantity to available if possible
            if ($availableQuantity > 0) {
                $quantity = $availableQuantity;
            } else {
                return "Insufficient stock for Product ID {$productId}";
            }
        }

        // Add to cart
        $result = $cart->updateQty($quantity, $productId, $productAttributeId);

        if ($result === true || $result > 0) {
            return true;
        }

        return "Failed to add Product ID {$productId} to cart";
    }

    /**
     * Build cart data array from Cart object
     */
    private function buildCartDataFromCart($cart)
    {
        $products = $cart->getProducts(true);
        $cartProducts = [];

        foreach ($products as $product) {
            $cartProducts[] = [
                'id' => (string) $product['id_product'],
                'id_product_attribute' => (string) (isset($product['id_product_attribute']) ? $product['id_product_attribute'] : 0),
                'quantity' => $product['cart_quantity']
            ];
        }

        return [
            'customer_id' => $cart->id_customer,
            'guest_id' => $cart->id_guest,
            'cart_contents' => [
                'products' => $cartProducts
            ]
        ];
    }

}
