<?php

require_once dirname(__FILE__) . '/../OpenappAbstractRESTController.php';

use Symfony\Component\HttpFoundation\Request;

/**
 * This REST endpoint recalculates basket pricing (VIRTUAL - no DB modifications)
 * Based on selected delivery method and discount codes
 *
 * Like WooCommerce, this endpoint creates a temporary cart for calculations
 * and never persists anything to the database.
 */
class ps_openappbasket_recalculateModuleFrontController extends OpenappAbstractRESTController
{
    /**
     * @var string Country ISO Code
     */
    private $countryIsoCode = 'PL';

    protected function processPostRequest()
    {
        header('Content-Type: application/json; charset=utf-8');

        $bodyRaw = Tools::file_get_contents('php://input');
        $_POST = json_decode($bodyRaw, true);

        $request = Request::createFromGlobals();
        $headers = $request->headers->all();

        $context = 'openapp_basket_recalculate';
        $this->ct_custom_log("POST /basket_recalculate called", $context);
        $this->ct_custom_log("Body: " . $bodyRaw, $context);

        // Verify HMAC with body hash
        $bodyHash = hash('sha256', $bodyRaw, true);
        $responseBodyHashBase64 = base64_encode($bodyHash);
        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if (!$validHmac) {
            $this->sendError('invalid_auth', $this->trans('Unauthorized request', [], 'Modules.Openapp.Cart'), 403);
        }

        // Validate request structure (WP standard - flat structure)
        if (!is_array($_POST)) {
            $this->sendError('invalid_data', $this->trans('Invalid JSON structure', [], 'Modules.Openapp.Cart'), 400);
        }

        $basketData = $_POST;

        $basketId = isset($basketData['id']) ? $basketData['id'] : null;
        $products = isset($basketData['products']) ? $basketData['products'] : [];
        $selectedDeliveryOption = isset($basketData['selectedDeliveryOption']) ? $basketData['selectedDeliveryOption'] : null;
        $loggedUser = isset($basketData['loggedUser']) ? $basketData['loggedUser'] : '0_0';

        // Extract discount codes from price.discounts[] (WP standard)
        $discountCodes = [];
        if (isset($basketData['price']['discounts']) && is_array($basketData['price']['discounts'])) {
            foreach ($basketData['price']['discounts'] as $discount) {
                if (isset($discount['code'])) {
                    $discountCodes[] = $discount['code'];
                }
            }
        }

        $langId = (int) Configuration::get('PS_LANG_DEFAULT');

        // === CREATE TEMPORARY CART - NEVER PERSISTED TO DB ===
        // This mimics WooCommerce's approach of creating a fake cart for calculations
        $tempCart = new Cart();
        $tempCart->id_lang = $langId;
        $tempCart->id_currency = (int) Configuration::get('PS_CURRENCY_DEFAULT');
        $tempCart->id_shop = (int) Context::getContext()->shop->id;
        $tempCart->id_shop_group = (int) Context::getContext()->shop->id_shop_group;
        // IMPORTANT: Do NOT call $tempCart->add() - this keeps it in memory only

        // Build products array and add to temp cart
        $outputProducts = [];
        $basketValueFloat = 0;

        foreach ($products as $product) {
            if (!isset($product['id']) || !isset($product['quantity'])) {
                continue;
            }

            // Parse product ID (format: "productId_attributeId")
            list($productId, $productAttributeId) = $this->parseProductId($product['id']);
            $quantity = (int) $product['quantity'];

            if ($productId <= 0 || $quantity <= 0) {
                continue;
            }

            // Validate product exists and is active
            $productObj = new Product($productId, true, $langId);
            if (!Validate::isLoadedObject($productObj) || !$productObj->active) {
                continue;
            }

            // Stock/availability checks aligned with WP behavior
            $error = null;
            $availableQuantity = StockAvailable::getQuantityAvailableByProduct($productId, $productAttributeId);
            if ($availableQuantity <= 0) {
                $error = 'OUT_OF_STOCK';
                $quantity = 0;
            } elseif ($quantity > $availableQuantity) {
                $quantity = $availableQuantity;
                $error = 'QUANTITY_TOO_BIG';
            }

            // Get current price (with taxes)
            $unitPrice = Product::getPriceStatic($productId, true, $productAttributeId);
            $linePrice = $unitPrice * $quantity;
            $basketValueFloat += $linePrice;

            // Get product image
            $imageLink = '';
            if ($productAttributeId > 0) {
                $combinationImages = $productObj->getCombinationImages($langId);
                if (isset($combinationImages[$productAttributeId]) && count($combinationImages[$productAttributeId])) {
                    $imageId = $combinationImages[$productAttributeId][0]['id_image'];
                    $imageLink = Context::getContext()->link->getImageLink($productObj->link_rewrite, $imageId, 'home_default');
                }
            }
            if ($imageLink == '') {
                $cover = Product::getCover($productId);
                if ($cover) {
                    $imageLink = Context::getContext()->link->getImageLink($productObj->link_rewrite, $cover['id_image'], 'home_default');
                }
            }

            // Get EAN (prefer combination EAN if available)
            $ean = $productObj->ean13;
            if ($productAttributeId > 0) {
                $combination = new Combination($productAttributeId);
                if (Validate::isLoadedObject($combination) && $combination->ean13) {
                    $ean = $combination->ean13;
                }
            }

            // Build product name with attributes
            $productName = $productObj->name;
            if ($productAttributeId > 0) {
                $combination = new Combination($productAttributeId);
                if (Validate::isLoadedObject($combination)) {
                    $attributes = $combination->getAttributesName($langId);
                    $attributesString = implode(', ', array_column($attributes, 'name'));
                    $productName .= " - " . $attributesString;
                }
            }

            $outputProducts[] = [
                'ean' => $ean,
                'id' => $this->formatProductId($productId, $productAttributeId),
                'name' => $productName,
                'images' => [$imageLink],
                'quantity' => $quantity,
                'unitPrice' => $this->groszeFormat($unitPrice),
                'linePrice' => $this->groszeFormat($linePrice),
                'originalUnitPrice' => $this->groszeFormat($unitPrice),
                'originalLinePrice' => $this->groszeFormat($linePrice)
            ];
            if ($error !== null) {
                $outputProducts[count($outputProducts) - 1]['error'] = $error;
            }
        }

        // Validate and calculate discounts (WITHOUT persisting to any cart)
        $appliedDiscounts = [];
        $totalDiscountValue = 0;

        foreach ($discountCodes as $code) {
            $discountInfo = $this->validateDiscountCode($code, $basketValueFloat);
            if ($discountInfo['valid']) {
                $appliedDiscounts[] = [
                    'code' => $code,
                    'value' => $discountInfo['value']
                ];
                $totalDiscountValue += $discountInfo['value'];
            }
        }

        // Calculate final basket value (in grosze/cents)
        $basketValueCents = $this->groszeFormat($basketValueFloat);
        $finalBasketValue = max(0, $basketValueCents - $totalDiscountValue);

        // Get shipping options (calculated manually since we don't have a real cart)
        $deliveryOptions = [];
        $deliveryCost = 0;
        $countryId = Country::getByIso($this->countryIsoCode);

        if ($countryId) {
            $country = new Country($countryId);
            $zoneId = $country->id_zone;
            $cartTotalPrice = $finalBasketValue / 100; // Convert cents back to float for shipping calc

            // Get all carriers and calculate shipping manually
            $carriers = Carrier::getCarriers($langId, true, false, false, null, Carrier::ALL_CARRIERS);

            foreach ($carriers as $carrier) {
                $mappedKey = Configuration::get('CARRIER_MAP_OA_' . $carrier['id_carrier']);

                if ($mappedKey !== false && !empty($mappedKey)) {
                    // Calculate shipping cost manually using Carrier object
                    $carrierObj = new Carrier($carrier['id_carrier']);
                    $shippingCost = 0;

                    // Handle free shipping carriers
                    if ($carrier['is_free'] == 1) {
                        $shippingCost = 0;
                    } else {
                        // Get shipping cost by price (most common method)
                        $shippingCost = $carrierObj->getDeliveryPriceByPrice($cartTotalPrice, $zoneId);

                        // If no price-based rate, check if there's a default
                        if ($shippingCost === false) {
                            $shippingCost = 0;
                        }
                    }

                    $shippingCostCents = $this->groszeFormat($shippingCost);

                    $deliveryOptions[] = [
                        'key' => $mappedKey,
                        'cost' => $shippingCostCents
                    ];
                }
            }

            // Get cost of selected delivery option
            if ($selectedDeliveryOption) {
                foreach ($deliveryOptions as $option) {
                    if ($option['key'] === $selectedDeliveryOption) {
                        $deliveryCost = $option['cost'];
                        break;
                    }
                }
            }
        }

        // Build response - temp cart is discarded after this (garbage collected)
        $response = [
            'id' => $basketId ?: 'virtual',
            'expiresAt' => gmdate('Y-m-d\TH:i:s\Z', strtotime('+1 day')),
            'price' => [
                'currency' => $this->getCurrency(),
                'discounts' => $appliedDiscounts,
                'basketValue' => $finalBasketValue
            ],
            'deliveryOptions' => $deliveryOptions,
            'products' => $outputProducts,
            'loggedUser' => $loggedUser
        ];

        if ($selectedDeliveryOption) {
            $response['selectedDeliveryOption'] = $selectedDeliveryOption;
        }

        $this->ct_custom_log("Response: " . json_encode($response), $context);

        // Calculate X-Server-Authorization
        $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
        if ($expectedXServerAuth !== null) {
            header('X-Server-Authorization: ' . $expectedXServerAuth);
        }

        http_response_code(200);
        $this->ajaxRender(json_encode($response));
        die;
    }

    /**
     * Validate discount code WITHOUT adding to any cart
     * Just checks if the code is valid and calculates its value
     *
     * @param string $discountCode The discount code to validate
     * @param float $cartTotalFloat The cart total in float (not cents)
     * @return array ['valid' => bool, 'value' => int (grosze/cents)]
     */
    private function validateDiscountCode($discountCode, $cartTotalFloat)
    {
        $result = [
            'valid' => false,
            'value' => 0
        ];

        // Find cart rule by code
        $cartRuleId = CartRule::getIdByCode($discountCode);
        if (!$cartRuleId) {
            return $result;
        }

        $cartRule = new CartRule($cartRuleId);
        if (!Validate::isLoadedObject($cartRule) || !$cartRule->active) {
            return $result;
        }

        // Check minimum amount requirement
        if ($cartRule->minimum_amount > 0 && $cartTotalFloat < $cartRule->minimum_amount) {
            return $result;
        }

        // Calculate discount value (without actually applying it)
        $discountValue = 0;

        if ($cartRule->reduction_percent > 0) {
            // Percentage discount
            $discountValue = $cartTotalFloat * ($cartRule->reduction_percent / 100);
        } elseif ($cartRule->reduction_amount > 0) {
            // Fixed amount discount
            $discountValue = $cartRule->reduction_amount;
        }

        $result['valid'] = true;
        $result['value'] = $this->groszeFormat($discountValue);

        return $result;
    }
}
