<?php
require_once dirname(__FILE__) . '/../OpenappAbstractRESTController.php';

use Symfony\Component\HttpFoundation\Request;

/**
 * This REST endpoint handles order returns from OpenApp
 *
 * POST /openapp/order-return
 */
    class ps_openapporderreturnModuleFrontController extends OpenappAbstractRESTController
    {
        protected function processPostRequest()
        {
            $bodyRaw = Tools::file_get_contents('php://input');
            $_POST = json_decode($bodyRaw, true);

            $oaOrderId = Tools::getValue('oaOrderId');
            $oaOrderReturnId = Tools::getValue('oaOrderReturnId');
            $shopOrderIdRaw = Tools::getValue('shopOrderId');
            $returnedProducts = Tools::getValue('returnedProducts', array());

            $request = Request::createFromGlobals();
            $headers = $request->headers->all();

        /**
         * Log body request
         */
        $this->ct_custom_log("orderReturn_body:");
        $json_post_data = json_encode($_POST, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $this->ct_custom_log($json_post_data);

        /**
         * Set default response headers
         */
        header('Content-Type: application/json; charset=utf-8');
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
        header('Expires: Thu, 01 Jan 1970 00:00:00 GMT');

        /**
         * Validate HMAC
         */
        $bodyHash = hash('sha256', $bodyRaw, true);
        $responseBodyHashBase64 = base64_encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);
        if (!$validHmac) {
            $this->sendError('invalid_auth', $this->trans('Unauthorized request', [], 'Modules.Openapp.Errors'), 403);
        }

        /**
         * Validate required fields
         */
        if (empty($oaOrderId)) {
            $this->sendError('missing_oaOrderId', $this->trans('OA Order ID is required', [], 'Modules.Openapp.Errors'), 400);
        }

            if (empty($oaOrderReturnId) || empty($shopOrderIdRaw)) {
                $this->sendError('missing_required_fields', $this->trans('Required fields missing', [], 'Modules.Openapp.Errors'), 400);
            }

            /**
             * Find order by id or reference (accept both)
             */
            $order = null;
            if (ctype_digit((string) $shopOrderIdRaw)) {
                $order = new Order((int) $shopOrderIdRaw);
            }

            if (!$order || !Validate::isLoadedObject($order)) {
                $orders = Order::getByReference($shopOrderIdRaw);
                if ($orders && $orders->count() > 0) {
                    $order = $orders->getFirst();
                }
            }

            if (!$order || !Validate::isLoadedObject($order)) {
                $this->sendError('order_not_found', $this->trans('Order not found', [], 'Modules.Openapp.Errors'), 404);
            }

            /**
             * Verify the order's transaction_id matches oaOrderId
         */
        $orderPayments = $order->getOrderPaymentCollection();
        $oaOrderIdMatch = false;

        if ($orderPayments && $orderPayments->count() > 0) {
            foreach ($orderPayments as $payment) {
                if ($payment->transaction_id === $oaOrderId) {
                    $oaOrderIdMatch = true;
                    break;
                }
            }
        }

        if (!$oaOrderIdMatch) {
            $this->sendError('order_mismatch', $this->trans('Order ID mismatch', [], 'Modules.Openapp.Errors'), 400);
        }

        /**
         * Generate case ID
         */
        $caseId = 'OA-RET-' . $order->id . '-' . date('ymd_His');

        /**
         * Create OrderReturn in PrestaShop
         */
            $orderReturn = new OrderReturn();
            $orderReturn->id_customer = (int) $order->id_customer;
            $orderReturn->id_order = (int) $order->id;
            $orderReturn->state = $this->getReceivedReturnStateId(); // Waiting for confirmation (received)
            $orderReturn->question = $this->buildReturnQuestion($caseId, $oaOrderReturnId, $returnedProducts);

            if (!$orderReturn->add()) {
                $this->sendError('return_creation_failed', $this->trans('Failed to create return request', [], 'Modules.Openapp.Errors'), 500);
            }

            /**
             * Add OrderReturnDetail for each returned product
             */
            if (!empty($returnedProducts) && is_array($returnedProducts)) {
                foreach ($returnedProducts as $returnedProduct) {
                    $this->addReturnDetail($orderReturn, $order, $returnedProduct);
                }
            }

            /**
             * Add order note with return details
             */
            $this->addOrderMessage($order, $caseId, $oaOrderId, $oaOrderReturnId, $returnedProducts);

            /**
             * Store payload snapshot in a private message for later parsing
             */
            $this->addPayloadMessage($order, $_POST);

            /**
             * Update persistent cart table with return info (optional tracking)
             */
            $this->updatePersistentCartWithReturn($order->id_cart, $caseId);

        /**
         * Success response
         */
        $response = [
            'caseId' => $caseId
        ];

        $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);

        if ($expectedXServerAuth !== null) {
            header('X-Server-Authorization: ' . $expectedXServerAuth);
        }

        $this->ct_custom_log("Order return processed successfully. Case ID: " . $caseId);

        $this->ajaxRender($this->encodeJsonResponse($response));
        die;
    }

        private function buildReturnQuestion($caseId, $oaReturnId, $returnedProducts)
        {
            $types = array();
            if (!empty($returnedProducts) && is_array($returnedProducts)) {
                foreach ($returnedProducts as $product) {
                    if (isset($product['returnType'])) {
                        $types[] = $product['returnType'];
                    }
                }
            }
            $types = array_unique($types);
            $typeString = implode(',', $types);

            return sprintf(
                'OA Return | caseId: %s | oaReturnId: %s | type: %s',
                $caseId,
                $oaReturnId,
                $typeString
            );
        }

        /**
         * Build return note text
         */
        private function buildReturnNote($order, $oaOrderId, $oaOrderReturnId, $returnedProducts)
        {
            $note = "=== OPENAPP RETURN REQUEST ===\n";
            $note .= "OA Order ID: " . $oaOrderId . "\n";
            $note .= "OA Return ID: " . $oaOrderReturnId . "\n\n";

            if (!empty($returnedProducts) && is_array($returnedProducts)) {
                $note .= "Returned Products:\n";
                foreach ($returnedProducts as $index => $product) {
                    $productId = $product['id'] ?? ($product['productId'] ?? null);
                    $qty = $product['returnedQuantity'] ?? ($product['quantity'] ?? 1);
                    $type = $product['returnType'] ?? ($product['type'] ?? 'N/A');
                    $reason = $product['reasonLabels']['pl'] ?? ($product['reason'] ?? 'N/A');
                    $linePrice = isset($product['linePrice']) ? Tools::displayPrice($product['linePrice'] / 100, new Currency($order->id_currency)) : '';

                    $displayName = $productId;
                    if ($productId) {
                        $displayName = $this->resolveProductName($order, $productId);
                    }

                    $note .= ($index + 1) . ". ";
                    $note .= "Product: " . $displayName . ", ";
                    $note .= "Qty: " . $qty . ", ";
                    $note .= "Type: " . $type . ", ";
                    $note .= "Reason: " . $reason;
                    if (!empty($linePrice)) {
                        $note .= ", Line: " . $linePrice;
                    }
                    $note .= "\n";
                }
            }

            return $note;
        }

        /**
         * Add return detail for a product
         */
        private function addReturnDetail($orderReturn, $order, $returnedProduct)
        {
            $productIdRaw = $returnedProduct['id'] ?? ($returnedProduct['productId'] ?? null);

            if (!$productIdRaw) {
                return false;
            }

            // Parse product ID and attribute ID (format: "123_0" or just "123")
            $parts = explode('_', $productIdRaw);
            $idProduct = (int) $parts[0];
            $idProductAttribute = isset($parts[1]) ? (int) $parts[1] : 0;

            $quantity = (int) ($returnedProduct['returnedQuantity'] ?? ($returnedProduct['quantity'] ?? 1));

            // Find the order detail for this product
            $orderDetails = OrderDetail::getList($order->id);
            $idOrderDetail = null;

            foreach ($orderDetails as $detail) {
                if ((int) $detail['product_id'] === $idProduct &&
                    (int) $detail['product_attribute_id'] === $idProductAttribute) {
                    $idOrderDetail = (int) $detail['id_order_detail'];
                    break;
                }
            }

            if (!$idOrderDetail) {
                $this->ct_custom_log("Could not find order detail for product: " . $productIdRaw);
                return false;
            }

            // Insert into order_return_detail table
            $sql = 'INSERT INTO `' . _DB_PREFIX_ . 'order_return_detail`
                    (`id_order_return`, `id_order_detail`, `id_customization`, `product_quantity`)
                    VALUES (' . (int) $orderReturn->id . ', ' . (int) $idOrderDetail . ', 0, ' . (int) $quantity . ')';

            return Db::getInstance()->execute($sql);
        }

        /**
         * Add message to order history
         */
        private function addOrderMessage($order, $caseId, $oaOrderId, $oaOrderReturnId, $returnedProducts)
        {
            $message = "=== OPENAPP RETURN REQUEST ===\n";
            $message .= "Case ID: " . $caseId . "\n";
            $message .= "OA Order ID: " . $oaOrderId . "\n";
            $message .= "OA Return ID: " . $oaOrderReturnId . "\n\n";

            if (!empty($returnedProducts) && is_array($returnedProducts)) {
                $message .= "Returned Products:\n";
                foreach ($returnedProducts as $index => $product) {
                    $productId = $product['id'] ?? ($product['productId'] ?? 'N/A');
                    $qty = $product['returnedQuantity'] ?? ($product['quantity'] ?? 1);
                    $type = $product['returnType'] ?? ($product['type'] ?? 'N/A');
                    $reason = $product['reasonLabels']['pl'] ?? ($product['reason'] ?? 'N/A');
                    $linePrice = isset($product['linePrice']) ? Tools::displayPrice($product['linePrice'] / 100, new Currency($order->id_currency)) : '';

                    $displayName = $productId;
                    if ($productId) {
                        $displayName = $this->resolveProductName($order, $productId);
                    }

                    $message .= ($index + 1) . ". ";
                    $message .= "Product: " . $displayName . ", ";
                    $message .= "Qty: " . $qty . ", ";
                    $message .= "Type: " . $type . ", ";
                    $message .= "Reason: " . $reason;
                    if (!empty($linePrice)) {
                        $message .= ", Line: " . $linePrice;
                    }
                    if (!empty($product['notes'])) {
                        $message .= ", Notes: " . $product['notes'];
                    }
                    $message .= "\n";
                }
            }

            $this->addPrivateOrderMessage($order, $message);
        }

        private function resolveProductName($order, $productIdRaw)
        {
            $parts = explode('_', $productIdRaw);
            $idProduct = (int) $parts[0];
            $idProductAttribute = isset($parts[1]) ? (int) $parts[1] : 0;

            $orderDetails = OrderDetail::getList($order->id);
            foreach ($orderDetails as $detail) {
                if ((int) $detail['product_id'] === $idProduct &&
                    (int) $detail['product_attribute_id'] === $idProductAttribute) {
                    return $detail['product_name'];
                }
            }

            return $productIdRaw;
        }

        private function addPayloadMessage($order, $payload)
        {
            $msg = new Message();
            $msg->message = 'OA_RETURN_PAYLOAD:' . json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            $msg->id_order = (int) $order->id;
            $msg->id_customer = (int) $order->id_customer;
            $msg->id_employee = $this->context->employee && $this->context->employee->id ? (int) $this->context->employee->id : 0;
            $msg->id_cart = (int) $order->id_cart;
            $msg->private = 1;
            if (!$msg->add()) {
                $this->ct_custom_log('Failed to add return payload message: ' . Db::getInstance()->getMsgError());
            }
        }

        private function getReceivedReturnStateId()
        {
            $stateId = (int) Configuration::get('OA_RETURN_STATE_RECEIVED', 1);
            return $stateId > 0 ? $stateId : 1;
        }

    /**
     * Update persistent cart table with return tracking (optional)
     */
    private function updatePersistentCartWithReturn($cartId, $caseId)
    {
        $table_name = _DB_PREFIX_ . 'ps_oa_persistent_cart';

        // Check if record exists
        $sql = new DbQuery();
        $sql->select('*');
        $sql->from('ps_oa_persistent_cart');
        $sql->where('cart_id = ' . (int) $cartId);
        $result = Db::getInstance()->executeS($sql);

        if ($result && count($result) > 0) {
            // Could add a return_case_id column in the future
            // For now, just log it
            $this->ct_custom_log("Return case " . $caseId . " associated with cart " . $cartId);
        }
    }
}
